const WebSocket = require('ws');
const http = require('http');
const mysql = require('mysql2/promise');
const { exec } = require('child_process');

// Configuration
const WS_PORT = process.env.WS_PORT || 8080;
const DB_CONFIG = {
  host: process.env.DB_HOST || 'localhost',
  user: process.env.DB_USER || 'root',
  password: process.env.DB_PASSWORD || '',
  database: process.env.DB_NAME || 'vortex',
};

// Store active connections
const clients = new Map(); // Map<userId, Set<WebSocket>>
const userSockets = new Map(); // Map<WebSocket, userId>
const tradeSubscriptions = new Map(); // Map<tradeId, Set<WebSocket>>

// Create HTTP server
const server = http.createServer();

// Create WebSocket server
const wss = new WebSocket.Server({ 
  server,
  path: '/ws',
  perMessageDeflate: false,
});

// Database connection pool
let dbPool;

// Initialize database connection
async function initDatabase() {
  try {
    dbPool = mysql.createPool({
      ...DB_CONFIG,
      waitForConnections: true,
      connectionLimit: 10,
      queueLimit: 0,
    });
    console.log('✅ Database connection pool created');
  } catch (error) {
    console.error('❌ Database connection error:', error);
  }
}

// Send notification to user
async function sendNotificationToUser(userId, notification) {
  const userConnections = clients.get(userId);
  if (userConnections && userConnections.size > 0) {
    const message = JSON.stringify({
      type: 'notification',
      data: notification,
    });
    
    userConnections.forEach((ws) => {
      if (ws.readyState === WebSocket.OPEN) {
        ws.send(message);
      }
    });
    return true;
  }
  return false;
}

// Send message to trade participants
async function sendMessageToTrade(tradeId, message) {
  const tradeConnections = tradeSubscriptions.get(tradeId);
  if (tradeConnections && tradeConnections.size > 0) {
    const messageStr = JSON.stringify({
      type: 'trade_message',
      tradeId: tradeId,
      data: message,
    });
    
    tradeConnections.forEach((ws) => {
      if (ws.readyState === WebSocket.OPEN) {
        ws.send(messageStr);
      }
    });
    return true;
  }
  return false;
}

// Broadcast transaction notification
async function broadcastTransaction(userId, transactionData) {
  await sendNotificationToUser(userId, {
    type: 'transaction',
    title: getTransactionTitle(transactionData.type),
    message: getTransactionMessage(transactionData),
    data: transactionData,
    timestamp: new Date().toISOString(),
  });
}

// Get transaction title
function getTransactionTitle(type) {
  const titles = {
    'send': 'Transfer Sent',
    'withdraw': 'Withdrawal Processed',
    'deposit': 'Deposit Received',
    'swap': 'Swap Completed',
    'exchange': 'Exchange Completed',
    'p2p': 'P2P Trade Update',
    'investment': 'Investment Processed',
  };
  return titles[type] || 'Transaction Update';
}

// Get transaction message
function getTransactionMessage(data) {
  const { type, amount, coin, status } = data;
  const statusText = status === 'Completed' ? 'completed' : status.toLowerCase();
  
  switch (type) {
    case 'send':
      return `You sent ${amount} ${coin} - ${statusText}`;
    case 'withdraw':
      return `Withdrawal of ${amount} ${coin} - ${statusText}`;
    case 'deposit':
      return `You received ${amount} ${coin} - ${statusText}`;
    case 'swap':
      return `Swapped ${data.fromAmount} ${data.fromCoin} for ${data.toAmount} ${data.toCoin}`;
    case 'exchange':
      return `Exchanged ${data.fromAmount} ${data.fromCoin} for ${data.toAmount} ${data.toCoin}`;
    case 'p2p':
      return `P2P trade ${statusText}: ${amount} ${coin}`;
    case 'investment':
      return `Investment of ${amount} ${coin} in ${data.tierName || 'tier'} - ${statusText}`;
    default:
      return `Transaction ${statusText}`;
  }
}

// Handle WebSocket connection
wss.on('connection', (ws, req) => {
  console.log('🔌 New WebSocket connection');
  
  let userId = null;
  let isAuthenticated = false;

  // Send welcome message
  ws.send(JSON.stringify({
    type: 'connection',
    status: 'connected',
    message: 'WebSocket connected. Please authenticate.',
  }));

  // Handle incoming messages
  ws.on('message', async (data) => {
    try {
      const message = JSON.parse(data.toString());
      
      switch (message.type) {
        case 'authenticate':
          await handleAuthentication(ws, message);
          break;
        
        case 'subscribe_trade':
          if (isAuthenticated) {
            handleSubscribeTrade(ws, message.tradeId);
          }
          break;
        
        case 'unsubscribe_trade':
          if (isAuthenticated) {
            handleUnsubscribeTrade(ws, message.tradeId);
          }
          break;
        
        case 'ping':
          ws.send(JSON.stringify({ type: 'pong' }));
          break;
        
        default:
          console.log('Unknown message type:', message.type);
      }
    } catch (error) {
      console.error('Error handling message:', error);
      ws.send(JSON.stringify({
        type: 'error',
        message: 'Invalid message format',
      }));
    }
  });

  // Handle authentication
  async function handleAuthentication(ws, message) {
    try {
      const { username, token } = message;
      
      if (!username) {
        ws.send(JSON.stringify({
          type: 'auth_error',
          message: 'Username required',
        }));
        return;
      }

      // Verify user exists in database
      const [users] = await dbPool.execute(
        'SELECT id, username FROM users WHERE username = ? OR email = ?',
        [username, username]
      );

      if (users.length === 0) {
        ws.send(JSON.stringify({
          type: 'auth_error',
          message: 'User not found',
        }));
        return;
      }

      userId = users[0].id;
      isAuthenticated = true;
      userSockets.set(ws, userId);

      // Add to user's connection set
      if (!clients.has(userId)) {
        clients.set(userId, new Set());
      }
      clients.get(userId).add(ws);

      console.log(`✅ User authenticated: ${username} (ID: ${userId})`);

      ws.send(JSON.stringify({
        type: 'authenticated',
        userId: userId,
        username: users[0].username,
        message: 'Authentication successful',
      }));
    } catch (error) {
      console.error('Authentication error:', error);
      ws.send(JSON.stringify({
        type: 'auth_error',
        message: 'Authentication failed',
      }));
    }
  }

  // Handle trade subscription
  function handleSubscribeTrade(ws, tradeId) {
    if (!tradeSubscriptions.has(tradeId)) {
      tradeSubscriptions.set(tradeId, new Set());
    }
    tradeSubscriptions.get(tradeId).add(ws);
    console.log(`📌 User ${userId} subscribed to trade ${tradeId}`);
  }

  // Handle trade unsubscription
  function handleUnsubscribeTrade(ws, tradeId) {
    const tradeConnections = tradeSubscriptions.get(tradeId);
    if (tradeConnections) {
      tradeConnections.delete(ws);
      if (tradeConnections.size === 0) {
        tradeSubscriptions.delete(tradeId);
      }
    }
  }

  // Handle disconnection
  ws.on('close', () => {
    console.log('🔌 WebSocket disconnected');
    
    if (userId) {
      const userConnections = clients.get(userId);
      if (userConnections) {
        userConnections.delete(ws);
        if (userConnections.size === 0) {
          clients.delete(userId);
        }
      }
    }

    userSockets.delete(ws);

    // Remove from all trade subscriptions
    tradeSubscriptions.forEach((connections, tradeId) => {
      connections.delete(ws);
      if (connections.size === 0) {
        tradeSubscriptions.delete(tradeId);
      }
    });
  });

  // Handle errors
  ws.on('error', (error) => {
    console.error('WebSocket error:', error);
  });

  // Heartbeat
  const heartbeat = setInterval(() => {
    if (ws.readyState === WebSocket.OPEN) {
      ws.send(JSON.stringify({ type: 'ping' }));
    } else {
      clearInterval(heartbeat);
    }
  }, 30000);
});

// HTTP endpoint to send notifications (called from PHP)
server.on('request', async (req, res) => {
  if (req.method === 'POST' && req.url === '/notify') {
    let body = '';
    req.on('data', chunk => { body += chunk.toString(); });
    req.on('end', async () => {
      try {
        const data = JSON.parse(body);
        const { userId, notification } = data;
        
        await sendNotificationToUser(userId, notification);
        
        res.writeHead(200, { 'Content-Type': 'application/json' });
        res.end(JSON.stringify({ success: true }));
      } catch (error) {
        res.writeHead(500, { 'Content-Type': 'application/json' });
        res.end(JSON.stringify({ success: false, error: error.message }));
      }
    });
  } else if (req.method === 'POST' && req.url === '/notify-trade') {
    let body = '';
    req.on('data', chunk => { body += chunk.toString(); });
    req.on('end', async () => {
      try {
        const data = JSON.parse(body);
        const { tradeId, message } = data;
        
        await sendMessageToTrade(tradeId, message);
        
        res.writeHead(200, { 'Content-Type': 'application/json' });
        res.end(JSON.stringify({ success: true }));
      } catch (error) {
        res.writeHead(500, { 'Content-Type': 'application/json' });
        res.end(JSON.stringify({ success: false, error: error.message }));
      }
    });
  } else {
    res.writeHead(404);
    res.end('Not found');
  }
});

// Start server
async function startServer() {
  await initDatabase();
  
  server.listen(WS_PORT, () => {
    console.log(`🚀 WebSocket server running on port ${WS_PORT}`);
    console.log(`📡 WebSocket endpoint: ws://localhost:${WS_PORT}/ws`);
  });
}

// Export functions for use in other modules
module.exports = {
  sendNotificationToUser,
  sendMessageToTrade,
  broadcastTransaction,
  startServer,
};

// Start server if run directly
if (require.main === module) {
  startServer().catch(console.error);
}

// Handle graceful shutdown
process.on('SIGTERM', () => {
  console.log('SIGTERM received, closing server...');
  wss.close(() => {
    server.close(() => {
      if (dbPool) {
        dbPool.end();
      }
      process.exit(0);
    });
  });
});

