# OxaPay Integration Guide

This integration allows users to deposit cryptocurrencies (excluding Solana) using OxaPay's static address service. Each user gets a unique static address per coin that remains constant for all deposits.

## Features

- ✅ Unique static addresses per user per coin
- ✅ Supports all featured coins (BTC, ETH, LTC, BNB, TRX, XRP, DOGE) except Solana
- ✅ Supports all non-Solana network coins
- ✅ Automatic deposit processing via webhooks
- ✅ QR code generation for easy deposits
- ✅ Secure address storage in database

## Setup Instructions

### 1. Get OxaPay Credentials

1. Sign up for an OxaPay merchant account at [https://oxapay.com](https://oxapay.com)
2. Navigate to your merchant dashboard
3. Get your **Merchant API Key** from the API settings section
4. Optionally set up a webhook secret for additional security

### 2. Configure OxaPay

Edit `backend/crispapi/oxapay/config.php` and add your credentials:

```php
define('OXAPAY_MERCHANT_API_KEY', 'YOUR_MERCHANT_API_KEY_HERE');
define('OXAPAY_TEST_MODE', false); // Set to true for testing
define('OXAPAY_WEBHOOK_SECRET', 'YOUR_WEBHOOK_SECRET_HERE'); // Optional
```

### 3. Run Database Migration

Execute the SQL migration script to create the necessary table:

```bash
mysql -u root -p vortex < backend/crispapi/oxapay/migration.sql
```

Or manually run the SQL in `backend/crispapi/oxapay/migration.sql` using phpMyAdmin or your database tool.

### 4. Configure Webhook URL

In your OxaPay merchant dashboard:
1. Go to Settings → Webhooks
2. Add webhook URL: `https://yourdomain.com/backend/crispapi/oxapay/webhook.php`
3. Enable webhook notifications for deposits

### 5. Update API Base URL

Make sure your frontend `config/env.ts` has the correct API URL:

```typescript
API_URL: process.env.API_URL || 'http://yourdomain.com/crispvort/backend'
```

## How It Works

### Address Creation Flow

1. User requests a deposit address for a coin (e.g., BTC, ETH)
2. System checks if user already has an address for that coin
3. If not, creates a new static address via OxaPay API
4. Address is stored in `oxapay_addresses` table
5. Same address is returned for all future requests

### Deposit Processing Flow

1. User sends cryptocurrency to their static address
2. OxaPay detects the deposit
3. OxaPay sends webhook notification to `webhook.php`
4. System processes the deposit:
   - Records in `payments` table
   - Updates user balance
   - Marks deposit as completed

## API Endpoints

### Create/Get Address

**Endpoint:** `POST /backend/crispapi/oxapay/create_address.php`

**Request:**
```json
{
  "username": "user123",
  "coin": "BTC",
  "cointicker": "BTC"
}
```

**Response:**
```json
{
  "success": true,
  "address": "bc1qxy2kgdygjrsqtzq2n0yrf2493p83kkfjhx0wlh",
  "message": "New address successfully created.",
  "is_new": true
}
```

### Webhook (Internal)

**Endpoint:** `POST /backend/crispapi/oxapay/webhook.php`

This endpoint is called by OxaPay when a deposit is detected. No manual calls needed.

## Supported Coins

The integration supports all OxaPay-supported cryptocurrencies, including:

- **Bitcoin (BTC)**
- **Ethereum (ETH)**
- **Litecoin (LTC)**
- **Binance Coin (BNB)**
- **Tron (TRX)**
- **Ripple (XRP)**
- **Dogecoin (DOGE)**
- **Tether (USDT)**
- **USD Coin (USDC)**
- And many more...

**Note:** Solana (SOL) deposits use native Solana addresses, not OxaPay.

## Frontend Integration

### Using OxaPayDeposit Component

```tsx
import OxaPayDeposit from '../components/OxaPayDeposit';

<OxaPayDeposit
  coinSymbol="BTC"
  coinName="Bitcoin"
  onClose={() => setModalVisible(false)}
/>
```

### Featured Coins Integration

The `FeaturedCoins` component now supports long-press to open deposit modal:
- Long press on any featured coin (except SOL) to open OxaPay deposit
- Regular press navigates to token details

## Database Schema

### oxapay_addresses Table

```sql
CREATE TABLE oxapay_addresses (
  id INT AUTO_INCREMENT PRIMARY KEY,
  username VARCHAR(256) NOT NULL,
  coin_symbol VARCHAR(10) NOT NULL,
  address VARCHAR(255) NOT NULL,
  order_id VARCHAR(255),
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  UNIQUE KEY unique_user_coin (username, coin_symbol)
);
```

## Security Considerations

1. **API Key Security**: Never commit your API key to version control
2. **Webhook Verification**: Implement signature verification if OxaPay provides it
3. **HTTPS**: Always use HTTPS for webhook endpoints in production
4. **Input Validation**: All user inputs are validated before API calls
5. **Error Handling**: Errors are logged but not exposed to users

## Testing

### Test Mode

Set `OXAPAY_TEST_MODE` to `true` in `config.php` for testing:

```php
define('OXAPAY_TEST_MODE', true);
```

### Test Address Creation

```bash
curl -X POST http://localhost/crispvort/backend/crispapi/oxapay/create_address.php \
  -H "Content-Type: application/json" \
  -d '{
    "username": "testuser",
    "coin": "BTC",
    "cointicker": "BTC"
  }'
```

## Troubleshooting

### Address Not Created

1. Check OxaPay API key is correct
2. Verify database connection
3. Check PHP error logs: `backend/crispapi/oxapay/error_log`
4. Ensure user exists in database

### Deposits Not Processing

1. Verify webhook URL is accessible
2. Check webhook.php logs
3. Ensure webhook is enabled in OxaPay dashboard
4. Verify database permissions

### Frontend Errors

1. Check API URL in `config/env.ts`
2. Verify user is logged in (`userData.username` exists)
3. Check browser console for errors
4. Verify CORS headers are set correctly

## Support

For OxaPay API documentation, visit: [https://docs.oxapay.com](https://docs.oxapay.com)

For issues with this integration, check:
- PHP error logs
- Webhook logs
- Database connection status
- OxaPay merchant dashboard

## License

This integration is part of the CrispVort project.

