<?php

class OxaPayService {
    private $merchantApiKey;
    private $apiUrl = 'https://api.oxapay.com/v1';
    private $db;

    public function __construct($merchantApiKey, $db) {
        $this->merchantApiKey = $merchantApiKey;
        $this->db = $db;
    }

    /**
     * Create a static address for a user and coin
     * @param string $username
     * @param string $coinSymbol (BTC, ETH, LTC, BNB, TRX, XRP, DOGE, etc.)
     * @param string $network Network name (Bitcoin, Ethereum, Litecoin, etc.)
     * @return array
     */
    public function createStaticAddress($username, $coinSymbol, $network) {
        try {
            // Check if user already has an address for this coin
            $existingAddress = $this->getUserAddress($username, $coinSymbol);
            if ($existingAddress) {
                return [
                    'success' => true,
                    'address' => $existingAddress,
                    'message' => 'Address already exists for this coin',
                    'is_new' => false
                ];
            }

            // Get user info
            $user = $this->getUserByUsername($username);
            if (!$user) {
                return [
                    'success' => false,
                    'message' => 'User not found'
                ];
            }

            // Map coin symbols to OxaPay network names
            $networkMap = $this->getNetworkMap();
            $oxapayNetwork = $networkMap[$coinSymbol] ?? $network;

            // Prepare callback URL
            $callbackUrl = $this->getCallbackUrl();

            // Create order ID
            $orderId = "{$username}_{$coinSymbol}_" . time();

            // Make API request to OxaPay
            // Using the correct endpoint for static address creation
            $url = $this->apiUrl . '/payment/static-address';
            
            // OxaPay API format for static address creation
            // Note: 'to_currency' is for auto-conversion feature, not for static addresses
            // For static addresses, we use 'currency' parameter
            $data = [
                'network' => $oxapayNetwork,
                'currency' => $coinSymbol, // Use 'currency' instead of 'to_currency'
                'callback_url' => $callbackUrl,
                'email' => $user['email'] ?? '',
                'order_id' => $orderId,
                'description' => "Deposit address for {$username} - {$coinSymbol}"
            ];

            $response = $this->makeApiRequest($url, $data);

            // Handle different response formats
            $address = null;
            if ($response['success']) {
                // Try different possible response structures
                $responseData = $response['data'];
                
                if (isset($responseData['address'])) {
                    $address = $responseData['address'];
                } elseif (isset($responseData['wallet'])) {
                    $address = $responseData['wallet'];
                } elseif (isset($responseData['data']['address'])) {
                    $address = $responseData['data']['address'];
                } elseif (isset($responseData['data']['wallet'])) {
                    $address = $responseData['data']['wallet'];
                } elseif (is_array($responseData) && isset($responseData[0]['address'])) {
                    $address = $responseData[0]['address'];
                } elseif (is_string($responseData) && strlen($responseData) > 20) {
                    // If data is directly the address string
                    $address = $responseData;
                }
            }

            if ($address) {
                // Store address in database
                $this->saveUserAddress($username, $coinSymbol, $address, $orderId);
                
                return [
                    'success' => true,
                    'address' => $address,
                    'order_id' => $orderId,
                    'message' => 'Address created successfully',
                    'is_new' => true
                ];
            } else {
                // Log the full response for debugging
                error_log("OxaPay Address Creation Failed. Full response: " . json_encode($response));
                return [
                    'success' => false,
                    'message' => $response['message'] ?? 'Failed to create address. Address not found in API response.',
                    'debug' => $response['response'] ?? null
                ];
            }
        } catch (Exception $e) {
            error_log("OxaPay Error: " . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Error creating address: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Get user's address for a specific coin
     */
    private function getUserAddress($username, $coinSymbol) {
        $coinColumn = strtolower($coinSymbol) . 'oxapay_address';
        
        // Check if column exists, if not use alternative table
        $stmt = $this->db->prepare("
            SELECT {$coinColumn} FROM users WHERE username = ? AND {$coinColumn} IS NOT NULL AND {$coinColumn} != ''
        ");
        
        try {
            $stmt->execute([$username]);
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            return $result ? $result[$coinColumn] : null;
        } catch (PDOException $e) {
            // Column might not exist, check oxapay_addresses table
            return $this->getAddressFromTable($username, $coinSymbol);
        }
    }

    /**
     * Get address from oxapay_addresses table
     */
    private function getAddressFromTable($username, $coinSymbol) {
        $stmt = $this->db->prepare("
            SELECT address FROM oxapay_addresses 
            WHERE username = ? AND coin_symbol = ?
        ");
        $stmt->execute([$username, $coinSymbol]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        return $result ? $result['address'] : null;
    }

    /**
     * Save user address to database
     */
    private function saveUserAddress($username, $coinSymbol, $address, $orderId) {
        // Try to save in oxapay_addresses table first
        $stmt = $this->db->prepare("
            INSERT INTO oxapay_addresses (username, coin_symbol, address, order_id, created_at)
            VALUES (?, ?, ?, ?, NOW())
            ON DUPLICATE KEY UPDATE address = ?, order_id = ?, updated_at = NOW()
        ");
        
        try {
            $stmt->execute([$username, $coinSymbol, $address, $orderId, $address, $orderId]);
        } catch (PDOException $e) {
            // Table might not exist, create it
            $this->createOxapayAddressesTable();
            $stmt->execute([$username, $coinSymbol, $address, $orderId, $address, $orderId]);
        }

        // Also update users table if column exists
        $coinColumn = strtolower($coinSymbol) . 'oxapay_address';
        try {
            $updateStmt = $this->db->prepare("
                UPDATE users SET {$coinColumn} = ? WHERE username = ?
            ");
            $updateStmt->execute([$address, $username]);
        } catch (PDOException $e) {
            // Column doesn't exist, that's okay
        }
    }

    /**
     * Create oxapay_addresses table if it doesn't exist
     */
    private function createOxapayAddressesTable() {
        $sql = "
            CREATE TABLE IF NOT EXISTS oxapay_addresses (
                id INT AUTO_INCREMENT PRIMARY KEY,
                username VARCHAR(256) NOT NULL,
                coin_symbol VARCHAR(10) NOT NULL,
                address VARCHAR(255) NOT NULL,
                order_id VARCHAR(255),
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                UNIQUE KEY unique_user_coin (username, coin_symbol),
                INDEX idx_username (username),
                INDEX idx_address (address)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4
        ";
        $this->db->exec($sql);
    }

    /**
     * Get user by username
     */
    private function getUserByUsername($username) {
        $stmt = $this->db->prepare("SELECT * FROM users WHERE username = ?");
        $stmt->execute([$username]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    /**
     * Map coin symbols to OxaPay network names
     */
    private function getNetworkMap() {
        return [
            'BTC' => 'Bitcoin',
            'ETH' => 'Ethereum',
            'LTC' => 'Litecoin',
            'BNB' => 'BinanceSmartChain',
            'TRX' => 'Tron',
            'XRP' => 'Ripple',
            'DOGE' => 'Dogecoin',
            'USDT' => 'Tether',
            'USDC' => 'USDCoin',
            'BUSD' => 'BinanceUSD',
            'SOL' => 'Solana',
            'ADA' => 'Cardano',
            'DOT' => 'Polkadot',
            'MATIC' => 'Polygon',
            'AVAX' => 'Avalanche'
        ];
    }

    /**
     * Make API request to OxaPay
     */
    private function makeApiRequest($url, $data) {
        $ch = curl_init($url);
        
        // OxaPay requires merchant_api_key in the header, not in the body
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => json_encode($data),
            CURLOPT_HTTPHEADER => [
                'merchant_api_key: ' . $this->merchantApiKey,
                'Content-Type: application/json',
                'Accept: application/json'
            ],
            CURLOPT_SSL_VERIFYPEER => true,
            CURLOPT_TIMEOUT => 30
        ]);

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);

        // Log the request and response for debugging
        error_log("OxaPay API Request URL: " . $url);
        error_log("OxaPay API Request Data: " . json_encode($data));
        error_log("OxaPay API Response Code: " . $httpCode);
        error_log("OxaPay API Response: " . $response);

        if ($error) {
            error_log("OxaPay CURL Error: " . $error);
            return [
                'success' => false,
                'message' => 'CURL Error: ' . $error
            ];
        }

        $responseData = json_decode($response, true);

        // Check if response is valid JSON
        if (json_last_error() !== JSON_ERROR_NONE) {
            error_log("OxaPay JSON Decode Error: " . json_last_error_msg());
            return [
                'success' => false,
                'message' => 'Invalid JSON response from API: ' . substr($response, 0, 200)
            ];
        }

        // OxaPay API response format:
        // Success: {"status": 200, "message": "Operation completed successfully!", "data": {...}, "error": []}
        // Error: {"status": 400, "message": "...", "error": {...}, "data": []}
        
        if ($httpCode === 200) {
            // Check if status is 200 in response body (OxaPay format)
            if (isset($responseData['status']) && $responseData['status'] == 200) {
                // Check if error array is empty (success) or has items (error)
                $hasErrors = isset($responseData['error']) && 
                            (is_array($responseData['error']) ? count($responseData['error']) > 0 : !empty($responseData['error']));
                
                if (!$hasErrors && isset($responseData['data'])) {
                    return [
                        'success' => true,
                        'data' => $responseData['data']
                    ];
                } elseif (!$hasErrors && isset($responseData['address'])) {
                    // Direct address in response
                    return [
                        'success' => true,
                        'data' => $responseData
                    ];
                } else {
                    // Has errors even though status is 200
                    $errorMsg = $responseData['message'] ?? 'Unknown API error';
                    $errorData = is_array($responseData['error']) && count($responseData['error']) > 0 
                                ? $responseData['error'][0] 
                                : $responseData['error'];
                    
                    return [
                        'success' => false,
                        'message' => $errorMsg,
                        'response' => $responseData,
                        'http_code' => $httpCode
                    ];
                }
            } elseif (isset($responseData['result']) && $responseData['result'] == 100) {
                // Alternative success format (result code 100)
                return [
                    'success' => true,
                    'data' => $responseData['data'] ?? $responseData
                ];
            } elseif (isset($responseData['status']) && $responseData['status'] === 'success') {
                // String status success
                return [
                    'success' => true,
                    'data' => $responseData['data'] ?? $responseData
                ];
            } elseif (isset($responseData['address'])) {
                // Direct address in response
                return [
                    'success' => true,
                    'data' => $responseData
                ];
            } else {
                // Check for error message
                $errorMsg = $responseData['message'] ?? $responseData['error'] ?? 'Unknown API error';
                $errorCode = $responseData['result'] ?? $responseData['code'] ?? $responseData['status'] ?? 'N/A';
                
                return [
                    'success' => false,
                    'message' => $errorMsg . ' (Code: ' . $errorCode . ')',
                    'response' => $responseData,
                    'http_code' => $httpCode
                ];
            }
        } else {
            $errorMsg = $responseData['message'] ?? $responseData['error'] ?? 'HTTP Error: ' . $httpCode;
            return [
                'success' => false,
                'message' => $errorMsg,
                'response' => $responseData,
                'http_code' => $httpCode
            ];
        }
    }

    /**
     * Get callback URL for webhooks
     */
    private function getCallbackUrl() {
        // Adjust this to your actual domain
        $protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http';
        $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
        return "{$protocol}://{$host}/backend/crispapi/oxapay/webhook.php";
    }

    /**
     * Verify webhook signature
     */
    public function verifyWebhook($data, $signature) {
        $calculatedSignature = hash_hmac('sha256', json_encode($data), $this->merchantApiKey);
        return hash_equals($calculatedSignature, $signature);
    }

    /**
     * Process deposit from webhook
     */
    public function processDeposit($webhookData) {
        try {
            $orderId = $webhookData['order_id'] ?? '';
            $amount = $webhookData['amount'] ?? 0;
            $currency = $webhookData['currency'] ?? '';
            $txid = $webhookData['txid'] ?? '';
            $status = $webhookData['status'] ?? 'pending';
            $address = $webhookData['address'] ?? '';

            // Extract username and coin from order_id
            if (preg_match('/^(.+)_([A-Z]+)_\d+$/', $orderId, $matches)) {
                $username = $matches[1];
                $coinSymbol = $matches[2];
            } else {
                // Fallback: find by address
                $addressInfo = $this->findAddressInfo($address);
                if (!$addressInfo) {
                    return ['success' => false, 'message' => 'Address not found'];
                }
                $username = $addressInfo['username'];
                $coinSymbol = $addressInfo['coin_symbol'];
            }

            // Record deposit in payments table
            $stmt = $this->db->prepare("
                INSERT INTO payments (username, txid, status, value, fin, address, coin, payment_id)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?)
            ");
            
            $paymentStatus = ($status === 'paid' || $status === 'completed') ? 'Completed' : 'Pending';
            $fin = ($status === 'paid' || $status === 'completed') ? 1 : 0;
            
            $stmt->execute([
                $username,
                $txid,
                $paymentStatus,
                $amount,
                $fin,
                $address,
                $coinSymbol,
                $orderId
            ]);

            // Update user balance if deposit is completed
            if ($fin === 1) {
                $this->updateUserBalance($username, $coinSymbol, $amount);
                
                // Save transaction to transactions table
                $this->saveDepositTransaction($username, $coinSymbol, $amount, $txid, $address);
                
                // Send real-time notification
                require_once __DIR__ . '/../notification_helper.php';
                $userId = $this->getUserIdFromUsername($username);
                if ($userId) {
                    sendNotification($userId, $username, [
                        'type' => 'transaction',
                        'title' => 'Deposit Received',
                        'message' => "You received {$amount} {$coinSymbol}",
                        'data' => [
                            'type' => 'deposit',
                            'amount' => $amount,
                            'coin' => $coinSymbol,
                            'status' => 'Completed',
                            'txnid' => $txid,
                        ],
                        'timestamp' => date('c'),
                    ]);
                }
            }

            return [
                'success' => true,
                'message' => 'Deposit processed',
                'username' => $username,
                'coin' => $coinSymbol,
                'amount' => $amount
            ];
        } catch (Exception $e) {
            error_log("OxaPay Deposit Processing Error: " . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Error processing deposit: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Find address info from database
     */
    private function findAddressInfo($address) {
        $stmt = $this->db->prepare("
            SELECT username, coin_symbol FROM oxapay_addresses WHERE address = ?
        ");
        $stmt->execute([$address]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    /**
     * Update user balance
     */
    private function updateUserBalance($username, $coinSymbol, $amount) {
        $coinColumn = 'v_' . strtolower($coinSymbol);
        
        try {
            $stmt = $this->db->prepare("
                UPDATE users SET {$coinColumn} = {$coinColumn} + ? WHERE username = ?
            ");
            $stmt->execute([$amount, $username]);
        } catch (PDOException $e) {
            // Column might not exist, try alternative
            error_log("Balance update error: " . $e->getMessage());
        }
    }

    /**
     * Save deposit transaction to transactions table
     */
    private function getUserIdFromUsername($username) {
        $stmt = $this->db->prepare("SELECT id FROM users WHERE username = ? OR email = ? LIMIT 1");
        $stmt->execute([$username, $username]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        return $user ? $user['id'] : null;
    }

    private function saveDepositTransaction($username, $coinSymbol, $amount, $txid, $address) {
        try {
            // Connect to vortex database for transactions table
            $transactions_db = new PDO("mysql:host=localhost;dbname=vortex", "root", "");
            $transactions_db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
            
            // Generate transaction ID if not provided
            if (empty($txid)) {
                $txid = $this->generateRandomString(13);
            }
            
            $coinName = $coinSymbol; // Use symbol as coin name
            $coinTicker = strtoupper($coinSymbol);
            
            $stmt = $transactions_db->prepare("
                INSERT INTO transactions (username, address, destination, amount, to_amount, status, type, coin, cointicker, txtype, txnid, fee)
                VALUES (?, ?, ?, ?, ?, 'Completed', 'Received', ?, ?, 'deposit', ?, '0.00000')
            ");
            
            $stmt->execute([
                $username,
                $address,
                'deposit',
                $amount,
                $amount,
                $coinName,
                $coinTicker,
                $txid
            ]);
        } catch (PDOException $e) {
            error_log("Error saving deposit transaction: " . $e->getMessage());
        }
    }

    /**
     * Generate random string for transaction ID
     */
    private function generateRandomString($length = 17) {
        $characters = '0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ';
        $charactersLength = strlen($characters);
        $randomString = '';
        for ($i = 0; $i < $length; $i++) {
            $randomString .= 'URI'.$characters[random_int(0, $charactersLength - 1)];
        }
        return $randomString;
    }
}

