<?php
// Enable error reporting for debugging (remove in production)
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: POST, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type");

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// Database connection details
$servername = "localhost";
$db_username = "root";
$password = "";
$dbname = "vortex";

// Create connection
$conn = new mysqli($servername, $db_username, $password, $dbname);

// Check connection
if ($conn->connect_error) {
    error_log("Database connection error: " . $conn->connect_error);
    http_response_code(500);
    echo json_encode(array("success" => false, "error" => "Database connection failed: " . $conn->connect_error));
    exit;
}

// Set charset to utf8
$conn->set_charset("utf8");

$raw_input = file_get_contents("php://input");
$data = json_decode($raw_input, true);

// Check if JSON decode was successful
if (json_last_error() !== JSON_ERROR_NONE) {
    error_log("JSON decode error: " . json_last_error_msg() . " - Input: " . $raw_input);
    http_response_code(400);
    echo json_encode(array("success" => false, "error" => "Invalid JSON data: " . json_last_error_msg()));
    $conn->close();
    exit;
}

// Retrieve data from the request
$username = isset($data['username']) ? $conn->real_escape_string(trim($data['username'])) : '';
$tier_id = intval($data['tier_id'] ?? 0);
$amount = floatval($data['amount'] ?? 0); // USD equivalent
$coin_amount = floatval($data['coin_amount'] ?? 0); // Coin amount
$coin = strtoupper($conn->real_escape_string($data['coin'] ?? 'USDT')); // Selected coin
$coin_price = floatval($data['coin_price'] ?? 1); // Coin price at time of investment

// Debug logging
error_log("Investment request - Username: " . $username . ", Tier: " . $tier_id . ", Amount: " . $amount . ", Coin: " . $coin . ", Coin Amount: " . $coin_amount);

if (empty($username) || $tier_id <= 0 || $amount <= 0 || $coin_amount <= 0) {
    echo json_encode(array(
        "success" => false, 
        "error" => "Missing required fields",
        "debug" => array(
            "username" => $username,
            "tier_id" => $tier_id,
            "amount" => $amount,
            "coin_amount" => $coin_amount
        )
    ));
    $conn->close();
    exit;
}

// Define tier information
$tiers = [
    1 => ['name' => 'Starter', 'min' => 10, 'max' => 99, 'apy' => 5.0],
    2 => ['name' => 'Bronze', 'min' => 100, 'max' => 499, 'apy' => 8.5],
    3 => ['name' => 'Silver', 'min' => 500, 'max' => 1999, 'apy' => 12.0],
    4 => ['name' => 'Gold', 'min' => 2000, 'max' => 9999, 'apy' => 15.5],
    5 => ['name' => 'Platinum', 'min' => 10000, 'max' => 49999, 'apy' => 18.0],
    6 => ['name' => 'Diamond', 'min' => 50000, 'max' => 999999, 'apy' => 22.0],
];

if (!isset($tiers[$tier_id])) {
    echo json_encode(array("success" => false, "error" => "Invalid tier ID"));
    $conn->close();
    exit;
}

$tier = $tiers[$tier_id];

// Validate amount range
if ($amount < $tier['min'] || $amount > $tier['max']) {
    echo json_encode(array("success" => false, "error" => "Amount must be between $" . $tier['min'] . " and $" . $tier['max']));
    $conn->close();
    exit;
}

// Generate transaction ID
function generateRandomString($length = 17) {
    $characters = '0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ';
    $charactersLength = strlen($characters);
    $randomString = '';
    for ($i = 0; $i < $length; $i++) {
        $randomString .= 'URI'.$characters[random_int(0, $charactersLength - 1)];
    }
    return $randomString;
}

$txnid = generateRandomString(13);

// Start transaction
$conn->begin_transaction();

try {
    // Check if investments table exists
    $table_check = $conn->query("SHOW TABLES LIKE 'investments'");
    if ($table_check->num_rows == 0) {
        throw new Exception("Investments table does not exist. Please run the investments.sql migration first.");
    }
    
    // Check if user exists
    $stmt = $conn->prepare("SELECT username FROM users WHERE username = ?");
    $stmt->bind_param("s", $username);
    $stmt->execute();
    $result = $stmt->get_result();
    $user = $result->fetch_assoc();

    if (!$user) {
        error_log("User not found in database - Username: " . $username);
        // Try to find user by email as fallback
        $stmt = $conn->prepare("SELECT username FROM users WHERE email = ?");
        $stmt->bind_param("s", $username);
        $stmt->execute();
        $result = $stmt->get_result();
        $user = $result->fetch_assoc();
        
        if ($user) {
            $username = $user['username']; // Update username to the correct one
            error_log("User found by email, using username: " . $username);
        } else {
            throw new Exception("User not found. Please check your username and try again.");
        }
    }

    // Check user's coin balance
    $coin_balance_key = 'v_' . strtolower($coin);
    $checkCoinColumn = $conn->query("SHOW COLUMNS FROM users LIKE '" . $conn->real_escape_string($coin_balance_key) . "'");
    
    if ($checkCoinColumn->num_rows == 0) {
        throw new Exception("Coin $coin is not supported. Column '$coin_balance_key' does not exist in users table.");
    }
    
    // Get current coin balance
    $stmt = $conn->prepare("SELECT `" . $coin_balance_key . "` FROM users WHERE username = ?");
    if (!$stmt) {
        throw new Exception("Failed to prepare coin balance check: " . $conn->error);
    }
    $stmt->bind_param("s", $username);
    if (!$stmt->execute()) {
        throw new Exception("Failed to execute coin balance check: " . $stmt->error);
    }
    $result = $stmt->get_result();
    $userCoinData = $result->fetch_assoc();
    $stmt->close();
    
    if (!$userCoinData) {
        throw new Exception("Failed to retrieve user coin balance data");
    }
    
    $currentCoinBalance = floatval($userCoinData[$coin_balance_key] ?? 0);
    
    // Check if user has sufficient coin balance
    if ($coin_amount > $currentCoinBalance) {
        throw new Exception("Insufficient $coin balance. Available: " . number_format($currentCoinBalance, 6) . " $coin");
    }
    
    error_log("User coin balance check - Coin: $coin, Balance: $currentCoinBalance, Required: $coin_amount");

    // Check if investment columns exist, if not use 0 as default
    $checkColumns = $conn->query("SHOW COLUMNS FROM users LIKE 'total_invested'");
    $hasTotalInvested = $checkColumns->num_rows > 0;
    
    $checkColumns2 = $conn->query("SHOW COLUMNS FROM users LIKE 'active_investments'");
    $hasActiveInvestments = $checkColumns2->num_rows > 0;

    $currentTotalInvested = 0;
    $currentActiveInvestments = 0;
    
    if ($hasTotalInvested || $hasActiveInvestments) {
        $selectFields = "username";
        if ($hasTotalInvested) $selectFields .= ", total_invested";
        if ($hasActiveInvestments) $selectFields .= ", active_investments";
        
        $stmt = $conn->prepare("SELECT " . $selectFields . " FROM users WHERE username = ?");
        $stmt->bind_param("s", $username);
        $stmt->execute();
        $result = $stmt->get_result();
        $userData = $result->fetch_assoc();
        
        if ($hasTotalInvested && isset($userData['total_invested'])) {
            $currentTotalInvested = floatval($userData['total_invested']);
        }
        if ($hasActiveInvestments && isset($userData['active_investments'])) {
            $currentActiveInvestments = floatval($userData['active_investments']);
        }
    }

    // Insert investment record
    // Check if coin columns exist in investments table
    $checkCoinColumns = $conn->query("SHOW COLUMNS FROM investments LIKE 'coin'");
    $hasCoinColumns = $checkCoinColumns->num_rows > 0;
    
    $tier_name = $tier['name'];
    $apy = $tier['apy'];
    
    if ($hasCoinColumns) {
        $stmt = $conn->prepare("
            INSERT INTO investments (username, tier_id, tier_name, amount, coin_amount, coin, coin_price, apy, status, created_at, txnid)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, 'active', NOW(), ?)
        ");
        if (!$stmt) {
            throw new Exception("Failed to prepare investment insert statement: " . $conn->error);
        }
        // Parameters: username(s), tier_id(i), tier_name(s), amount(d), coin_amount(d), coin(s), coin_price(d), apy(d), txnid(s)
        $stmt->bind_param("sisddsdds", $username, $tier_id, $tier_name, $amount, $coin_amount, $coin, $coin_price, $apy, $txnid);
    } else {
        $stmt = $conn->prepare("
            INSERT INTO investments (username, tier_id, tier_name, amount, apy, status, created_at, txnid)
            VALUES (?, ?, ?, ?, ?, 'active', NOW(), ?)
        ");
        if (!$stmt) {
            throw new Exception("Failed to prepare investment insert statement: " . $conn->error);
        }
        // Parameters: username(s), tier_id(i), tier_name(s), amount(d), apy(d), txnid(s)
        $stmt->bind_param("sisdds", $username, $tier_id, $tier_name, $amount, $apy, $txnid);
    }
    
    if (!$stmt->execute()) {
        throw new Exception("Failed to execute investment insert: " . $stmt->error);
    }
    
    $investment_id = $conn->insert_id;
    
    if ($investment_id <= 0) {
        throw new Exception("Failed to get investment ID after insert");
    }
    
    error_log("Investment record created - ID: " . $investment_id);

    // Update user's investment totals (only if columns exist)
    if ($hasTotalInvested || $hasActiveInvestments) {
        $updateFields = [];
        $updateValues = [];
        $updateTypes = "";
        
        if ($hasTotalInvested) {
            $newTotalInvested = $currentTotalInvested + $amount;
            $updateFields[] = "total_invested = ?";
            $updateValues[] = $newTotalInvested;
            $updateTypes .= "d";
        }
        
        if ($hasActiveInvestments) {
            $newActiveInvestments = $currentActiveInvestments + $amount;
            $updateFields[] = "active_investments = ?";
            $updateValues[] = $newActiveInvestments;
            $updateTypes .= "d";
        }
        
        $updateTypes .= "s"; // for username
        $updateValues[] = $username;
        
        $updateSql = "UPDATE users SET " . implode(", ", $updateFields) . " WHERE username = ?";
        $stmt = $conn->prepare($updateSql);
        $stmt->bind_param($updateTypes, ...$updateValues);
        $stmt->execute();
    }

    // Deduct coin from user balance
    $newCoinBalance = $currentCoinBalance - $coin_amount;
    $updateCoinSql = "UPDATE users SET " . $coin_balance_key . " = ? WHERE username = ?";
    $stmt = $conn->prepare($updateCoinSql);
    if (!$stmt) {
        throw new Exception("Failed to prepare coin balance update: " . $conn->error);
    }
    $stmt->bind_param("ds", $newCoinBalance, $username);
    if (!$stmt->execute()) {
        throw new Exception("Failed to update coin balance: " . $stmt->error);
    }
    error_log("Coin balance updated - New balance: " . $newCoinBalance . " " . $coin);

    // Record transaction in transactions table
    $stmt = $conn->prepare("
        INSERT INTO transactions (username, address, destination, amount, to_amount, status, type, coin, cointicker, txtype, txnid, fee)
        VALUES (?, 'investment', ?, ?, ?, 'Completed', 'Send', ?, ?, 'investment', ?, '0.00000')
    ");
    if (!$stmt) {
        throw new Exception("Failed to prepare transaction insert statement: " . $conn->error);
    }
    $destination = $tier_name . ' Tier';
    $to_amount = $coin_amount; // Store coin amount
    $transaction_coin = $coin; // Use the selected coin
    $cointicker = $coin;
    // Parameters: username(s), destination(s), amount(d), to_amount(d), coin(s), cointicker(s), txnid(s)
    // Note: txtype is hardcoded as 'investment' in the SQL, so we don't bind it
    $stmt->bind_param("ssddsss", $username, $destination, $coin_amount, $to_amount, $transaction_coin, $cointicker, $txnid);
    
    if (!$stmt->execute()) {
        error_log("Warning: Failed to insert transaction record: " . $stmt->error);
        // Don't throw exception here, as investment was successful
    }

    $conn->commit();
    
    error_log("Investment successful - Username: " . $username . ", Investment ID: " . $investment_id);
    
    // Send real-time notification
    require_once __DIR__ . '/notification_helper.php';
    if ($username) {
        $userId = getUserIdFromUsername($conn, $username);
        if ($userId) {
            sendNotification($userId, $username, [
                'type' => 'investment',
                'title' => 'Investment Processed',
                'message' => "You invested {$coin_amount} {$coin} (${$amount}) in {$tier_name} tier",
                'data' => [
                    'type' => 'investment',
                    'amount' => $amount,
                    'coin_amount' => $coin_amount,
                    'coin' => $coin,
                    'tierName' => $tier_name,
                    'apy' => $apy,
                    'status' => 'Completed',
                    'investment_id' => $investment_id,
                ],
                'timestamp' => date('c'),
            ]);
        }
    }
    
    $response = array(
        "success" => true,
        "message" => "Investment successful! You've invested $" . number_format($amount, 2) . " in " . $tier_name . " tier.",
        "investment_id" => $investment_id,
        "tier_name" => $tier_name,
        "amount" => $amount,
        "apy" => $apy
    );
    
    error_log("Sending response: " . json_encode($response));
    
    // Ensure we output the response correctly
    header("Content-Type: application/json; charset=UTF-8");
    echo json_encode($response);
    exit;
} catch (Exception $e) {
    if (isset($conn) && $conn->in_transaction) {
        $conn->rollback();
    }
    error_log("Investment error: " . $e->getMessage());
    error_log("Stack trace: " . $e->getTraceAsString());
    error_log("File: " . $e->getFile() . " Line: " . $e->getLine());
    http_response_code(500);
    header("Content-Type: application/json; charset=UTF-8");
    echo json_encode(array(
        "success" => false, 
        "error" => "Investment failed: " . $e->getMessage(),
        "debug" => array(
            "file" => $e->getFile(),
            "line" => $e->getLine()
        )
    ));
    if (isset($conn)) {
        $conn->close();
    }
    exit;
} catch (Error $e) {
    // Catch PHP 7+ errors (fatal errors, type errors, etc.)
    if (isset($conn) && $conn->in_transaction) {
        $conn->rollback();
    }
    error_log("Investment fatal error: " . $e->getMessage());
    error_log("Stack trace: " . $e->getTraceAsString());
    error_log("File: " . $e->getFile() . " Line: " . $e->getLine());
    http_response_code(500);
    header("Content-Type: application/json; charset=UTF-8");
    echo json_encode(array(
        "success" => false, 
        "error" => "Investment failed: " . $e->getMessage(),
        "debug" => array(
            "file" => $e->getFile(),
            "line" => $e->getLine()
        )
    ));
    if (isset($conn)) {
        $conn->close();
    }
    exit;
}

if (isset($conn)) {
    $conn->close();
}
?>

