<?php
/**
 * Exchange Order Books Handler
 * Handles order creation, matching, and trade execution
 * Saves trades to both transactions and exchange_trades tables
 */

// Set headers first
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type");

// Handle preflight OPTIONS request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// Wrap everything in try-catch to ensure we always return JSON
try {

// Enable error logging
ini_set('display_errors', 0);
ini_set('log_errors', 1);
error_reporting(E_ALL);

// Custom logging function
function logMessage($message, $data = null) {
    try {
        $logEntry = date("Y-m-d H:i:s") . " [ORDERBOOK] " . $message;
        if ($data !== null) {
            $logEntry .= " | Data: " . json_encode($data);
        }
        $logEntry .= PHP_EOL;
        error_log($logEntry);
        // Also log to a specific file
        $logFile = __DIR__ . '/orderbook_debug.log';
        @file_put_contents($logFile, $logEntry, FILE_APPEND);
    } catch (Exception $e) {
        // Silently fail logging to prevent breaking the script
        error_log("Logging error: " . $e->getMessage());
    }
}

// Set error handler to catch fatal errors
function handleFatalError() {
    $error = error_get_last();
    if ($error !== NULL && in_array($error['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR])) {
        http_response_code(500);
        header("Content-Type: application/json; charset=UTF-8");
        echo json_encode([
            "success" => false,
            "error" => "Fatal PHP Error",
            "message" => $error['message'],
            "file" => $error['file'],
            "line" => $error['line'],
            "type" => $error['type']
        ]);
        exit;
    }
}
register_shutdown_function('handleFatalError');

// Set exception handler
set_exception_handler(function($exception) {
    http_response_code(500);
    header("Content-Type: application/json; charset=UTF-8");
    echo json_encode([
        "success" => false,
        "error" => "Uncaught Exception",
        "message" => $exception->getMessage(),
        "file" => $exception->getFile(),
        "line" => $exception->getLine(),
        "trace" => $exception->getTraceAsString()
    ]);
    exit;
});

// Database connection
$servername = "localhost";
$username = "coinfzbe_pro_user";
$password = "_xS89yo#g?Rv";
$dbname = "coinfzbe_vortexprp";

$conn = new mysqli($servername, $username, $password, $dbname);

if ($conn->connect_error) {
    logMessage("Database connection failed", ['error' => $conn->connect_error]);
    http_response_code(500);
    echo json_encode(["error" => "Connection failed: " . $conn->connect_error]);
    exit;
}

logMessage("Database connection successful");

// Try to require notification helper, but don't fail if it doesn't exist
$notificationHelperPath = __DIR__ . '/notification_helper.php';
if (file_exists($notificationHelperPath)) {
    try {
        require_once $notificationHelperPath;
        logMessage("Notification helper loaded");
    } catch (Exception $e) {
        logMessage("Error loading notification helper", ['error' => $e->getMessage()]);
    }
} else {
    logMessage("Notification helper not found, continuing without it");
    // Define stub functions if helper doesn't exist
    if (!function_exists('sendNotification')) {
        function sendNotification($userId, $username, $notification) {
            return false;
        }
    }
    if (!function_exists('getUserIdFromUsername')) {
        function getUserIdFromUsername($conn, $username) {
            $username_escaped = $conn->real_escape_string($username);
            $result = $conn->query("SELECT id FROM users WHERE username = '$username_escaped' LIMIT 1");
            if ($result && $result->num_rows > 0) {
                $row = $result->fetch_assoc();
                return $row['id'];
            }
            return null;
        }
    }
}

// Get request data
logMessage("=== NEW REQUEST ===", [
    'method' => $_SERVER['REQUEST_METHOD'],
    'uri' => $_SERVER['REQUEST_URI'] ?? 'N/A'
]);

try {
    $rawInput = file_get_contents("php://input");
    logMessage("Raw input received", [
        'length' => strlen($rawInput),
        'preview' => substr($rawInput, 0, 200)
    ]);
    
    $data = json_decode($rawInput, true);
    
    if (json_last_error() !== JSON_ERROR_NONE) {
        logMessage("JSON decode error", [
            'error' => json_last_error_msg(),
            'code' => json_last_error(),
            'input' => substr($rawInput, 0, 200)
        ]);
        
        // If JSON decode fails, try to handle as GET request for get_orderbook
        if ($_SERVER['REQUEST_METHOD'] === 'GET' || empty($rawInput)) {
            $data = [
                'action' => 'get_orderbook',
                'pair' => $_GET['pair'] ?? 'BTC/USDT',
                'limit' => intval($_GET['limit'] ?? 20)
            ];
            logMessage("Using GET parameters", $data);
        } else {
            http_response_code(400);
            $errorMsg = "Invalid JSON: " . json_last_error_msg();
            logMessage("Returning error", ['error' => $errorMsg]);
            echo json_encode(["error" => $errorMsg]);
            exit;
        }
    } else {
        logMessage("JSON decoded successfully", $data);
    }
} catch (Exception $e) {
    logMessage("Exception in request parsing", [
        'error' => $e->getMessage(),
        'file' => $e->getFile(),
        'line' => $e->getLine()
    ]);
    http_response_code(400);
    echo json_encode(["error" => "Error parsing request: " . $e->getMessage()]);
    exit;
}

$action = $data['action'] ?? 'create_order'; // create_order, get_orderbook, cancel_order
logMessage("Action determined", ['action' => $action]);

// Ensure getUserIdFromUsername exists (it should be from notification_helper.php)
if (!function_exists('getUserIdFromUsername')) {
    function getUserIdFromUsername($conn, $username) {
        $username_escaped = $conn->real_escape_string($username);
        $result = $conn->query("SELECT id FROM users WHERE username = '$username_escaped' LIMIT 1");
        if ($result && $result->num_rows > 0) {
            $row = $result->fetch_assoc();
            return $row['id'];
        }
        return null;
    }
}

function generateRandomString($length = 17) {
    $characters = '0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ';
    $charactersLength = strlen($characters);
    $randomString = '';
    for ($i = 0; $i < $length; $i++) {
        $randomString .= 'URI'.$characters[random_int(0, $charactersLength - 1)];
    }
    return $randomString;
}

function getUserBalance($conn, $username, $coin) {
    $coin_lower = strtolower($coin);
    $username_escaped = $conn->real_escape_string($username);
    $result = $conn->query("SELECT v_{$coin_lower} FROM users WHERE username = '$username_escaped' LIMIT 1");
    if ($result && $result->num_rows > 0) {
        $row = $result->fetch_assoc();
        return floatval($row["v_{$coin_lower}"] ?? 0);
    }
    return 0;
}

function updateUserBalance($conn, $username, $coin, $amount, $operation = 'add') {
    $coin_lower = strtolower($coin);
    $username_escaped = $conn->real_escape_string($username);
    $amount_escaped = floatval($amount);
    
    if ($operation === 'add') {
        $sql = "UPDATE users SET v_{$coin_lower} = v_{$coin_lower} + {$amount_escaped} WHERE username = '$username_escaped'";
    } else {
        $sql = "UPDATE users SET v_{$coin_lower} = v_{$coin_lower} - {$amount_escaped} WHERE username = '$username_escaped'";
    }
    
    return $conn->query($sql);
}

function matchOrders($conn, $newOrder) {
    $matches = [];
    $remainingAmount = floatval($newOrder['remaining_amount']);
    
    if ($newOrder['order_type'] === 'buy') {
        // Find sell orders (asks) with price <= buy order price, sorted by price ASC (best first)
        $sql = "SELECT * FROM orders 
                WHERE pair = '{$newOrder['pair']}' 
                AND order_type = 'sell' 
                AND status IN ('pending', 'partially_filled')
                AND price <= {$newOrder['price']}
                ORDER BY price ASC, created_at ASC
                LIMIT 20";
    } else {
        // Find buy orders (bids) with price >= sell order price, sorted by price DESC (best first)
        $sql = "SELECT * FROM orders 
                WHERE pair = '{$newOrder['pair']}' 
                AND order_type = 'buy' 
                AND status IN ('pending', 'partially_filled')
                AND price >= {$newOrder['price']}
                ORDER BY price DESC, created_at ASC
                LIMIT 20";
    }
    
    $result = $conn->query($sql);
    
    if ($result && $result->num_rows > 0) {
        while (($row = $result->fetch_assoc()) && $remainingAmount > 0) {
            // Verify the matched order still has balance before matching
            $matchedOrderUsername = $row['username'];
            $matchedOrderType = $row['order_type'];
            
            if ($matchedOrderType === 'buy') {
                // Verify buyer has enough quote coin
                $quoteCoin = $row['quote_coin'];
                $neededQuote = floatval($row['remaining_amount']) * floatval($row['price']);
                $buyerBalance = getUserBalance($conn, $matchedOrderUsername, $quoteCoin);
                if ($buyerBalance < $neededQuote) {
                    logMessage("Skipping match - buyer insufficient balance", [
                        'buyer' => $matchedOrderUsername,
                        'needed' => $neededQuote,
                        'has' => $buyerBalance
                    ]);
                    continue; // Skip this order, try next
                }
            } else {
                // Verify seller has enough base coin
                $baseCoin = $row['base_coin'];
                $neededBase = floatval($row['remaining_amount']);
                $sellerBalance = getUserBalance($conn, $matchedOrderUsername, $baseCoin);
                if ($sellerBalance < $neededBase) {
                    logMessage("Skipping match - seller insufficient balance", [
                        'seller' => $matchedOrderUsername,
                        'needed' => $neededBase,
                        'has' => $sellerBalance
                    ]);
                    continue; // Skip this order, try next
                }
            }
            
            $matchAmount = min($remainingAmount, floatval($row['remaining_amount']));
            $matches[] = [
                'order' => $row,
                'amount' => $matchAmount
            ];
            $remainingAmount -= $matchAmount;
        }
    }
    
    return $matches;
}

function executeTrade($conn, $buyOrder, $sellOrder, $amount, $price, $tradingFee = 0.001) {
    $conn->autocommit(FALSE);
    
    try {
        logMessage("Executing trade", [
            'buy_order_id' => $buyOrder['id'],
            'sell_order_id' => $sellOrder['id'],
            'amount' => $amount,
            'price' => $price,
            'trading_fee' => $tradingFee
        ]);
        
        // Get user IDs
        $buyerId = getUserIdFromUsername($conn, $buyOrder['username']);
        $sellerId = getUserIdFromUsername($conn, $sellOrder['username']);
        
        if (!$buyerId || !$sellerId) {
            throw new Exception("User not found - Buyer: " . ($buyerId ? "OK" : "NOT FOUND") . ", Seller: " . ($sellerId ? "OK" : "NOT FOUND"));
        }
        
        $total = $price * $amount;
        $fee = $total * $tradingFee;
        $totalWithFee = $total + $fee;
        $tradeId = 'EX' . generateRandomString(15);
        
        // Update balances
        // Buyer: deduct quote coin (USDT) + fee, add base coin (BTC)
        $quoteCoin = $buyOrder['quote_coin'];
        $baseCoin = $buyOrder['base_coin'];
        
        // Verify buyer has enough quote coin before deducting (including fee)
        $buyerQuoteBalance = getUserBalance($conn, $buyOrder['username'], $quoteCoin);
        if ($buyerQuoteBalance < $totalWithFee) {
            throw new Exception("Buyer insufficient balance: Need {$totalWithFee} {$quoteCoin} (including {$fee} fee), have {$buyerQuoteBalance}");
        }
        
        // Verify seller has enough base coin before deducting
        $sellerBaseBalance = getUserBalance($conn, $sellOrder['username'], $baseCoin);
        if ($sellerBaseBalance < $amount) {
            throw new Exception("Seller insufficient balance: Need {$amount} {$baseCoin}, have {$sellerBaseBalance}");
        }
        
        logMessage("Balance checks passed", [
            'buyer_quote_balance' => $buyerQuoteBalance,
            'seller_base_balance' => $sellerBaseBalance,
            'fee' => $fee
        ]);
        
        // Buyer: deduct quote coin + fee
        if (!updateUserBalance($conn, $buyOrder['username'], $quoteCoin, $totalWithFee, 'subtract')) {
            throw new Exception("Failed to deduct buyer balance");
        }
        // Buyer: add base coin (full amount, fee already deducted from quote)
        if (!updateUserBalance($conn, $buyOrder['username'], $baseCoin, $amount, 'add')) {
            throw new Exception("Failed to credit buyer base coin");
        }
        
        // Seller: deduct base coin (BTC), add quote coin (USDT) - fee deducted from received amount
        $sellerReceives = $total - ($total * $tradingFee);
        if (!updateUserBalance($conn, $sellOrder['username'], $baseCoin, $amount, 'subtract')) {
            throw new Exception("Failed to deduct seller balance");
        }
        if (!updateUserBalance($conn, $sellOrder['username'], $quoteCoin, $sellerReceives, 'add')) {
            throw new Exception("Failed to credit seller quote coin");
        }
        
        logMessage("Balances updated successfully");
        
        // Save to exchange_trades table
        $totalFee = $fee + ($total * $tradingFee); // Buyer fee + seller fee
        $sqlTrade = "INSERT INTO exchange_trades (
            trade_id, buy_order_id, sell_order_id, buyer_id, buyer_username, 
            seller_id, seller_username, pair, base_coin, quote_coin, 
            price, amount, total, fee, created_at
        ) VALUES (
            '$tradeId', {$buyOrder['id']}, {$sellOrder['id']}, 
            $buyerId, '{$buyOrder['username']}', 
            $sellerId, '{$sellOrder['username']}', 
            '{$buyOrder['pair']}', '$baseCoin', '$quoteCoin', 
            $price, $amount, $total, $totalFee, NOW()
        )";
        
        if (!$conn->query($sqlTrade)) {
            throw new Exception("Failed to save trade: " . $conn->error);
        }
        
        // Save to transactions table for buyer
        $txid1 = generateRandomString(13);
        $coinNameMap = [
            'BTC' => 'Bitcoin', 'ETH' => 'Ethereum', 'SOL' => 'Solana',
            'BNB' => 'Binance Coin', 'USDT' => 'Tether', 'ADA' => 'Cardano'
        ];
        $baseCoinName = $coinNameMap[$baseCoin] ?? $baseCoin;
        $quoteCoinName = $coinNameMap[$quoteCoin] ?? $quoteCoin;
        
        // Buyer: Send transaction (deducted quote coin)
        $sqlTx1 = "INSERT INTO transactions (
            username, address, destination, amount, to_amount, status, 
            type, coin, cointicker, txtype, txnid, fee
        ) VALUES (
            '{$buyOrder['username']}', 'exchange', 'orderbook', $total, $total, 
            'Completed', 'Send', '$quoteCoinName', '$quoteCoin', 'exchange', 
            '$txid1', 0.00000
        )";
        
        // Buyer: Received transaction (received base coin)
        $txid2 = generateRandomString(13);
        $sqlTx2 = "INSERT INTO transactions (
            username, address, destination, amount, to_amount, status, 
            type, coin, cointicker, txtype, txnid, fee
        ) VALUES (
            '{$buyOrder['username']}', 'exchange', 'orderbook', $amount, $amount, 
            'Completed', 'Received', '$baseCoinName', '$baseCoin', 'exchange', 
            '$txid2', 0.00000
        )";
        
        // Seller: Send transaction (deducted base coin)
        $txid3 = generateRandomString(13);
        $sqlTx3 = "INSERT INTO transactions (
            username, address, destination, amount, to_amount, status, 
            type, coin, cointicker, txtype, txnid, fee
        ) VALUES (
            '{$sellOrder['username']}', 'exchange', 'orderbook', $amount, $amount, 
            'Completed', 'Send', '$baseCoinName', '$baseCoin', 'exchange', 
            '$txid3', 0.00000
        )";
        
        // Seller: Received transaction (received quote coin)
        $txid4 = generateRandomString(13);
        $sqlTx4 = "INSERT INTO transactions (
            username, address, destination, amount, to_amount, status, 
            type, coin, cointicker, txtype, txnid, fee
        ) VALUES (
            '{$sellOrder['username']}', 'exchange', 'orderbook', $total, $total, 
            'Completed', 'Received', '$quoteCoinName', '$quoteCoin', 'exchange', 
            '$txid4', 0.00000
        )";
        
        $conn->query($sqlTx1);
        $conn->query($sqlTx2);
        $conn->query($sqlTx3);
        $conn->query($sqlTx4);
        
        // Update order statuses
        $buyRemaining = floatval($buyOrder['remaining_amount']) - $amount;
        $sellRemaining = floatval($sellOrder['remaining_amount']) - $amount;
        
        $buyStatus = $buyRemaining <= 0 ? 'filled' : 'partially_filled';
        $sellStatus = $sellRemaining <= 0 ? 'filled' : 'partially_filled';
        
        $sqlUpdateBuy = "UPDATE orders SET 
            filled_amount = filled_amount + $amount,
            remaining_amount = $buyRemaining,
            status = '$buyStatus'
            WHERE id = {$buyOrder['id']}";
        
        $sqlUpdateSell = "UPDATE orders SET 
            filled_amount = filled_amount + $amount,
            remaining_amount = $sellRemaining,
            status = '$sellStatus'
            WHERE id = {$sellOrder['id']}";
        
        $conn->query($sqlUpdateBuy);
        $conn->query($sqlUpdateSell);
        
        // Send notifications
        if ($buyerId) {
            sendNotification($buyerId, $buyOrder['username'], [
                'type' => 'transaction',
                'title' => 'Trade Executed',
                'message' => "Bought {$amount} {$baseCoin} for {$total} {$quoteCoin}",
                'data' => ['type' => 'exchange_trade', 'trade_id' => $tradeId],
                'timestamp' => date('c'),
            ]);
        }
        
        if ($sellerId) {
            sendNotification($sellerId, $sellOrder['username'], [
                'type' => 'transaction',
                'title' => 'Trade Executed',
                'message' => "Sold {$amount} {$baseCoin} for {$total} {$quoteCoin}",
                'data' => ['type' => 'exchange_trade', 'trade_id' => $tradeId],
                'timestamp' => date('c'),
            ]);
        }
        
        $conn->commit();
        return ['success' => true, 'trade_id' => $tradeId];
        
    } catch (Exception $e) {
        $conn->rollback();
        return ['success' => false, 'error' => $e->getMessage()];
    } finally {
        $conn->autocommit(TRUE);
    }
}

// Handle different actions
if ($action === 'create_order') {
    $username = $data['username'] ?? '';
    $pair = $data['pair'] ?? ''; // e.g., BTC/USDT
    $orderType = $data['order_type'] ?? ''; // buy or sell
    $orderSide = $data['order_side'] ?? 'limit'; // limit or market
    $price = floatval($data['price'] ?? 0);
    $amount = floatval($data['amount'] ?? 0);
    
    if (!$username || !$pair || !$orderType || !$amount) {
        echo json_encode(["error" => "Missing required parameters"]);
        exit;
    }
    
    // Parse pair
    $pairParts = explode('/', $pair);
    if (count($pairParts) !== 2) {
        echo json_encode(["error" => "Invalid trading pair format"]);
        exit;
    }
    $baseCoin = $pairParts[0];
    $quoteCoin = $pairParts[1];
    
    // Get user ID
    $userId = getUserIdFromUsername($conn, $username);
    if (!$userId) {
        echo json_encode(["error" => "User not found"]);
        exit;
    }
    
    // For market orders, get best available price
    if ($orderSide === 'market') {
        if ($orderType === 'buy') {
            // Get best sell price
            $result = $conn->query("SELECT MIN(price) as best_price FROM orders WHERE pair = '$pair' AND order_type = 'sell' AND status IN ('pending', 'partially_filled')");
            if ($result && $result->num_rows > 0) {
                $row = $result->fetch_assoc();
                $price = floatval($row['best_price'] ?? $price);
            }
        } else {
            // Get best buy price
            $result = $conn->query("SELECT MAX(price) as best_price FROM orders WHERE pair = '$pair' AND order_type = 'buy' AND status IN ('pending', 'partially_filled')");
            if ($result && $result->num_rows > 0) {
                $row = $result->fetch_assoc();
                $price = floatval($row['best_price'] ?? $price);
            }
        }
    }
    
    if ($price <= 0) {
        echo json_encode(["error" => "Invalid price"]);
        exit;
    }
    
    // Get trading fee from settings
    $tradingFeeResult = $conn->query("SELECT trading_fee FROM settings LIMIT 1");
    $tradingFee = 0.001; // Default 0.1% if not found
    if ($tradingFeeResult && $tradingFeeResult->num_rows > 0) {
        $feeRow = $tradingFeeResult->fetch_assoc();
        $tradingFee = floatval($feeRow['trading_fee'] ?? 0.001);
    }
    
    $total = $price * $amount;
    $fee = $total * $tradingFee;
    $totalWithFee = $total + $fee;
    
    // Check balance (including fee for buy orders)
    if ($orderType === 'buy') {
        // Need quote coin (USDT) to buy + fee
        $balance = getUserBalance($conn, $username, $quoteCoin);
        if ($balance < $totalWithFee) {
            echo json_encode(["error" => "Insufficient {$quoteCoin} balance. Need {$totalWithFee} (including {$fee} fee), have {$balance}"]);
            exit;
        }
    } else {
        // Need base coin (BTC) to sell (fee deducted from received amount)
        $balance = getUserBalance($conn, $username, $baseCoin);
        if ($balance < $amount) {
            echo json_encode(["error" => "Insufficient {$baseCoin} balance. Need {$amount}, have {$balance}"]);
            exit;
        }
    }
    
    // For market orders, try immediate matching first
    if ($orderSide === 'market') {
        // Check for immediate matching opportunities
        $matchQuery = $orderType === 'buy' 
            ? "SELECT * FROM orders WHERE pair = '$pair' AND order_type = 'sell' AND status IN ('pending', 'partially_filled') ORDER BY price ASC, created_at ASC LIMIT 1"
            : "SELECT * FROM orders WHERE pair = '$pair' AND order_type = 'buy' AND status IN ('pending', 'partially_filled') ORDER BY price DESC, created_at ASC LIMIT 1";
        
        $matchResult = $conn->query($matchQuery);
        if ($matchResult && $matchResult->num_rows > 0) {
            $matchedOrderRow = $matchResult->fetch_assoc();
            $matchAmount = min($amount, floatval($matchedOrderRow['remaining_amount']));
            $executionPrice = floatval($matchedOrderRow['price']);
            
            // Execute trade immediately without creating order
            $tempOrder = [
                'id' => 0,
                'username' => $username,
                'pair' => $pair,
                'base_coin' => $baseCoin,
                'quote_coin' => $quoteCoin,
                'order_type' => $orderType,
                'price' => $executionPrice,
                'remaining_amount' => $matchAmount
            ];
            
            if ($orderType === 'buy') {
                $tradeResult = executeTrade($conn, $tempOrder, $matchedOrderRow, $matchAmount, $executionPrice, $tradingFee);
            } else {
                $tradeResult = executeTrade($conn, $matchedOrderRow, $tempOrder, $matchAmount, $executionPrice, $tradingFee);
            }
            
            if ($tradeResult['success']) {
                // If fully matched, return success without creating order
                if ($matchAmount >= $amount) {
                    echo json_encode([
                        "success" => true,
                        "message" => "Trade executed immediately",
                        "executed_trades" => 1,
                        "trades" => [$tradeResult],
                        "immediate_match" => true
                    ]);
                    exit;
                }
                // If partially matched, continue to create order for remaining amount
                $amount = $amount - $matchAmount;
                $total = $price * $amount;
                $fee = $total * $tradingFee;
                $totalWithFee = $total + $fee;
            }
        }
    }
    
    // Lock balance (deduct immediately for limit orders)
    if ($orderSide === 'limit') {
        if ($orderType === 'buy') {
            updateUserBalance($conn, $username, $quoteCoin, $totalWithFee, 'subtract');
        } else {
            updateUserBalance($conn, $username, $baseCoin, $amount, 'subtract');
        }
    }
    
    // Create order
    $sql = "INSERT INTO orders (
        user_id, username, pair, base_coin, quote_coin, order_type, order_side,
        price, amount, filled_amount, remaining_amount, total, status
    ) VALUES (
        $userId, '$username', '$pair', '$baseCoin', '$quoteCoin', '$orderType', '$orderSide',
        $price, $amount, 0, $amount, $total, 'pending'
    )";
    
    if ($conn->query($sql)) {
        $orderId = $conn->insert_id;
        
        // Try to match orders
        $newOrder = [
            'id' => $orderId,
            'username' => $username,
            'pair' => $pair,
            'order_type' => $orderType,
            'price' => $price,
            'remaining_amount' => $amount
        ];
        
        $matches = matchOrders($conn, $newOrder);
        $executedTrades = [];
        
        foreach ($matches as $match) {
            $matchedOrder = $match['order'];
            $matchAmount = $match['amount'];
            $executionPrice = $matchedOrder['price']; // Use matched order's price
            
            // Get full order details
            $result = $conn->query("SELECT * FROM orders WHERE id = {$newOrder['id']}");
            $fullNewOrder = $result->fetch_assoc();
            
            if ($orderType === 'buy') {
                $tradeResult = executeTrade($conn, $fullNewOrder, $matchedOrder, $matchAmount, $executionPrice, $tradingFee);
            } else {
                $tradeResult = executeTrade($conn, $matchedOrder, $fullNewOrder, $matchAmount, $executionPrice, $tradingFee);
            }
            
            if ($tradeResult['success']) {
                $executedTrades[] = $tradeResult;
                $newOrder['remaining_amount'] -= $matchAmount;
            }
        }
        
        // Update order status if partially or fully filled
        if ($newOrder['remaining_amount'] < $amount) {
            $newStatus = $newOrder['remaining_amount'] <= 0 ? 'filled' : 'partially_filled';
            $conn->query("UPDATE orders SET status = '$newStatus', remaining_amount = {$newOrder['remaining_amount']} WHERE id = {$newOrder['id']}");
        }
        
        echo json_encode([
            "success" => true,
            "order_id" => $orderId,
            "message" => "Order created successfully",
            "executed_trades" => count($executedTrades),
            "trades" => $executedTrades
        ]);
    } else {
        echo json_encode(["error" => "Failed to create order: " . $conn->error]);
    }
    
} elseif ($action === 'get_orderbook') {
    try {
        logMessage("Starting get_orderbook", ['pair' => $data['pair'] ?? 'BTC/USDT']);
        
        $pair = $data['pair'] ?? 'BTC/USDT';
        $limit = intval($data['limit'] ?? 20);
        
        logMessage("Parameters", ['pair' => $pair, 'limit' => $limit]);
        
        // Check if orders table exists
        logMessage("Checking if orders table exists");
        $tableCheck = $conn->query("SHOW TABLES LIKE 'orders'");
        if (!$tableCheck) {
            logMessage("Table check query failed", ['error' => $conn->error]);
            http_response_code(500);
            echo json_encode([
                "success" => false,
                "error" => "Database error checking tables: " . $conn->error
            ]);
            exit;
        }
        
        if ($tableCheck->num_rows === 0) {
            logMessage("Orders table does not exist");
            // Table doesn't exist, return empty order book
            echo json_encode([
                "success" => true,
                "pair" => $pair,
                "bids" => [],
                "asks" => [],
                "message" => "Order book tables not initialized. Please run migration first."
            ]);
            exit;
        }
        
        logMessage("Orders table exists, proceeding with queries");
        
        // Escape pair for SQL
        $pair_escaped = $conn->real_escape_string($pair);
        logMessage("Pair escaped", ['original' => $pair, 'escaped' => $pair_escaped]);
        
        // Get buy orders (bids) - sorted by price DESC
        $sqlBuy = "SELECT * FROM orders 
                   WHERE pair = '$pair_escaped' AND order_type = 'buy' AND status IN ('pending', 'partially_filled')
                   ORDER BY price DESC, created_at ASC 
                   LIMIT $limit";
        
        logMessage("Executing buy orders query", ['sql' => $sqlBuy]);
        
        // Get sell orders (asks) - sorted by price ASC
        $sqlSell = "SELECT * FROM orders 
                    WHERE pair = '$pair_escaped' AND order_type = 'sell' AND status IN ('pending', 'partially_filled')
                    ORDER BY price ASC, created_at ASC 
                    LIMIT $limit";
        
        logMessage("Executing sell orders query", ['sql' => $sqlSell]);
        
        $buyOrders = [];
        $sellOrders = [];
        
        $result = $conn->query($sqlBuy);
        if ($result) {
            logMessage("Buy orders query successful", ['num_rows' => $result->num_rows]);
            while ($row = $result->fetch_assoc()) {
                $buyOrder = [
                    'price' => floatval($row['price']),
                    'amount' => floatval($row['remaining_amount']),
                    'total' => floatval($row['remaining_amount']) * floatval($row['price'])
                ];
                $buyOrders[] = $buyOrder;
                logMessage("Processed buy order", $buyOrder);
            }
        } else {
            logMessage("Buy orders query failed", [
                'error' => $conn->error,
                'errno' => $conn->errno,
                'sql' => $sqlBuy
            ]);
        }
        
        $result = $conn->query($sqlSell);
        if ($result) {
            logMessage("Sell orders query successful", ['num_rows' => $result->num_rows]);
            while ($row = $result->fetch_assoc()) {
                $sellOrder = [
                    'price' => floatval($row['price']),
                    'amount' => floatval($row['remaining_amount']),
                    'total' => floatval($row['remaining_amount']) * floatval($row['price'])
                ];
                $sellOrders[] = $sellOrder;
                logMessage("Processed sell order", $sellOrder);
            }
        } else {
            logMessage("Sell orders query failed", [
                'error' => $conn->error,
                'errno' => $conn->errno,
                'sql' => $sqlSell
            ]);
        }
        
        $response = [
            "success" => true,
            "pair" => $pair,
            "bids" => $buyOrders,
            "asks" => $sellOrders
        ];
        
        logMessage("Returning response", [
            'bids_count' => count($buyOrders),
            'asks_count' => count($sellOrders)
        ]);
        
        $jsonResponse = json_encode($response);
        logMessage("Sending response", ['response_length' => strlen($jsonResponse)]);
        echo $jsonResponse;
        
    } catch (Exception $e) {
        logMessage("Exception in get_orderbook", [
            'error' => $e->getMessage(),
            'file' => $e->getFile(),
            'line' => $e->getLine(),
            'trace' => $e->getTraceAsString()
        ]);
        http_response_code(500);
        $errorResponse = [
            "success" => false,
            "error" => "Error fetching order book: " . $e->getMessage(),
            "details" => [
                "file" => $e->getFile(),
                "line" => $e->getLine()
            ]
        ];
        logMessage("Sending error response", $errorResponse);
        echo json_encode($errorResponse);
    }
    
} elseif ($action === 'get_user_orders') {
    $username = $data['username'] ?? '';
    $status = $data['status'] ?? null; // null = all, 'pending', 'partially_filled', 'filled', 'cancelled'
    
    if (!$username) {
        http_response_code(400);
        echo json_encode(["error" => "Username is required"]);
        exit;
    }
    
    $username_escaped = $conn->real_escape_string($username);
    $sql = "SELECT * FROM orders WHERE username = '$username_escaped'";
    
    if ($status) {
        $status_escaped = $conn->real_escape_string($status);
        $sql .= " AND status = '$status_escaped'";
    }
    
    $sql .= " ORDER BY created_at DESC LIMIT 100";
    
    $result = $conn->query($sql);
    $orders = [];
    
    if ($result) {
        while ($row = $result->fetch_assoc()) {
            $orders[] = [
                'id' => intval($row['id']),
                'pair' => $row['pair'],
                'order_type' => $row['order_type'],
                'order_side' => $row['order_side'],
                'price' => floatval($row['price']),
                'amount' => floatval($row['amount']),
                'filled_amount' => floatval($row['filled_amount']),
                'remaining_amount' => floatval($row['remaining_amount']),
                'total' => floatval($row['total']),
                'status' => $row['status'],
                'created_at' => $row['created_at'],
                'updated_at' => $row['updated_at'],
                'fill_percentage' => $row['amount'] > 0 ? (floatval($row['filled_amount']) / floatval($row['amount'])) * 100 : 0
            ];
        }
    }
    
    echo json_encode([
        "success" => true,
        "orders" => $orders,
        "count" => count($orders)
    ]);
    
} elseif ($action === 'get_user_trades') {
    $username = $data['username'] ?? '';
    $limit = intval($data['limit'] ?? 50);
    
    if (!$username) {
        http_response_code(400);
        echo json_encode(["error" => "Username is required"]);
        exit;
    }
    
    $username_escaped = $conn->real_escape_string($username);
    $sql = "SELECT * FROM exchange_trades 
            WHERE buyer_username = '$username_escaped' OR seller_username = '$username_escaped'
            ORDER BY created_at DESC LIMIT $limit";
    
    $result = $conn->query($sql);
    $trades = [];
    
    if ($result) {
        while ($row = $result->fetch_assoc()) {
            $isBuyer = $row['buyer_username'] === $username;
            $trades[] = [
                'id' => intval($row['id']),
                'trade_id' => $row['trade_id'],
                'pair' => $row['pair'],
                'side' => $isBuyer ? 'buy' : 'sell',
                'price' => floatval($row['price']),
                'amount' => floatval($row['amount']),
                'total' => floatval($row['total']),
                'fee' => floatval($row['fee']),
                'counterparty' => $isBuyer ? $row['seller_username'] : $row['buyer_username'],
                'created_at' => $row['created_at']
            ];
        }
    }
    
    echo json_encode([
        "success" => true,
        "trades" => $trades,
        "count" => count($trades)
    ]);
    
} elseif ($action === 'get_recent_trades') {
    $pair = $data['pair'] ?? 'BTC/USDT';
    $limit = intval($data['limit'] ?? 20);
    
    $pair_escaped = $conn->real_escape_string($pair);
    $sql = "SELECT * FROM exchange_trades 
            WHERE pair = '$pair_escaped'
            ORDER BY created_at DESC LIMIT $limit";
    
    $result = $conn->query($sql);
    $trades = [];
    
    if ($result) {
        while ($row = $result->fetch_assoc()) {
            $trades[] = [
                'id' => intval($row['id']),
                'price' => floatval($row['price']),
                'amount' => floatval($row['amount']),
                'total' => floatval($row['total']),
                'side' => 'buy', // Can determine from order IDs if needed
                'time' => $row['created_at']
            ];
        }
    }
    
    echo json_encode([
        "success" => true,
        "trades" => $trades
    ]);
    
} elseif ($action === 'cancel_order') {
    $orderId = intval($data['order_id'] ?? 0);
    $username = $data['username'] ?? '';
    
    if (!$orderId || !$username) {
        http_response_code(400);
        echo json_encode(["error" => "Missing order_id or username"]);
        exit;
    }
    
    $username_escaped = $conn->real_escape_string($username);
    
    // Get order
    $result = $conn->query("SELECT * FROM orders WHERE id = $orderId AND username = '$username_escaped'");
    if (!$result || $result->num_rows === 0) {
        http_response_code(404);
        echo json_encode(["error" => "Order not found"]);
        exit;
    }
    
    $order = $result->fetch_assoc();
    
    if ($order['status'] === 'filled' || $order['status'] === 'cancelled') {
        http_response_code(400);
        echo json_encode(["error" => "Order cannot be cancelled. Status: " . $order['status']]);
        exit;
    }
    
    $conn->autocommit(FALSE);
    
    try {
        // Refund remaining balance
        $remainingAmount = floatval($order['remaining_amount']);
        if ($order['order_type'] === 'buy') {
            // Refund quote coin
            $refundAmount = $remainingAmount * floatval($order['price']);
            if (!updateUserBalance($conn, $username, $order['quote_coin'], $refundAmount, 'add')) {
                throw new Exception("Failed to refund balance");
            }
        } else {
            // Refund base coin
            if (!updateUserBalance($conn, $username, $order['base_coin'], $remainingAmount, 'add')) {
                throw new Exception("Failed to refund balance");
            }
        }
        
        // Cancel order
        if (!$conn->query("UPDATE orders SET status = 'cancelled' WHERE id = $orderId")) {
            throw new Exception("Failed to cancel order: " . $conn->error);
        }
        
        $conn->commit();
        
        echo json_encode([
            "success" => true,
            "message" => "Order cancelled successfully"
        ]);
    } catch (Exception $e) {
        $conn->rollback();
        http_response_code(500);
        echo json_encode([
            "success" => false,
            "error" => $e->getMessage()
        ]);
    } finally {
        $conn->autocommit(TRUE);
    }
    
} else {
    http_response_code(400);
    echo json_encode(["error" => "Invalid action"]);
}

$conn->close();

} catch (Throwable $e) {
    // Catch any uncaught exceptions or errors
    http_response_code(500);
    header("Content-Type: application/json; charset=UTF-8");
    echo json_encode([
        "success" => false,
        "error" => "Internal Server Error",
        "message" => $e->getMessage(),
        "file" => $e->getFile(),
        "line" => $e->getLine(),
        "type" => get_class($e)
    ]);
    error_log("Fatal error in exchange_orderbooks.php: " . $e->getMessage() . " in " . $e->getFile() . " on line " . $e->getLine());
}
?>

