<?php
require_once __DIR__ . '/../config/database.php';
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: POST, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type");

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

header("Content-Type: application/json");

try {
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        throw new Exception('Only POST method allowed');
    }

    if (!isset($_POST['user_id']) || empty($_POST['user_id'])) {
        throw new Exception('User ID is required');
    }

    $userId = intval($_POST['user_id']);
    $idType = $_POST['id_type'] ?? '';
    $fullName = $_POST['full_name'] ?? '';
    $address = $_POST['address'] ?? '';
    $dateOfBirth = $_POST['date_of_birth'] ?? '';

    // Validate required fields
    if (empty($idType) || empty($fullName) || empty($address) || empty($dateOfBirth)) {
        throw new Exception('All fields are required');
    }

    if (!isset($_FILES['front_id']) || $_FILES['front_id']['error'] !== UPLOAD_ERR_OK) {
        throw new Exception('Front ID image is required');
    }

    if (!isset($_FILES['back_id']) || $_FILES['back_id']['error'] !== UPLOAD_ERR_OK) {
        throw new Exception('Back ID image is required');
    }

    $allowedTypes = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif', 'image/webp'];
    $maxSize = 10 * 1024 * 1024; // 10MB

    // Validate and upload front ID
    $frontFile = $_FILES['front_id'];
    $finfo = finfo_open(FILEINFO_MIME_TYPE);
    $frontMimeType = finfo_file($finfo, $frontFile['tmp_name']);
    
    if (!in_array($frontMimeType, $allowedTypes)) {
        throw new Exception('Front ID: Invalid file type. Only JPEG, PNG, GIF, and WebP are allowed.');
    }
    if ($frontFile['size'] > $maxSize) {
        throw new Exception('Front ID: File size exceeds 10MB limit.');
    }

    // Validate and upload back ID
    $backFile = $_FILES['back_id'];
    $backMimeType = finfo_file($finfo, $backFile['tmp_name']);
    
    if (!in_array($backMimeType, $allowedTypes)) {
        throw new Exception('Back ID: Invalid file type. Only JPEG, PNG, GIF, and WebP are allowed.');
    }
    if ($backFile['size'] > $maxSize) {
        throw new Exception('Back ID: File size exceeds 10MB limit.');
    }
    finfo_close($finfo);

    // Create uploads directory if it doesn't exist
    $uploadDir = __DIR__ . '/../uploads/kyc/';
    if (!file_exists($uploadDir)) {
        mkdir($uploadDir, 0777, true);
    }

    $timestamp = time();

    // Upload front ID
    $frontExtension = pathinfo($frontFile['name'], PATHINFO_EXTENSION);
    $frontFilename = 'kyc_front_' . $userId . '_' . $timestamp . '.' . $frontExtension;
    $frontFilepath = $uploadDir . $frontFilename;
    if (!move_uploaded_file($frontFile['tmp_name'], $frontFilepath)) {
        throw new Exception('Failed to save front ID image');
    }
    $frontUrl = '/crispvort/backend/uploads/kyc/' . $frontFilename;

    // Upload back ID
    $backExtension = pathinfo($backFile['name'], PATHINFO_EXTENSION);
    $backFilename = 'kyc_back_' . $userId . '_' . $timestamp . '.' . $backExtension;
    $backFilepath = $uploadDir . $backFilename;
    if (!move_uploaded_file($backFile['tmp_name'], $backFilepath)) {
        // Clean up front file if back fails
        if (file_exists($frontFilepath)) {
            unlink($frontFilepath);
        }
        throw new Exception('Failed to save back ID image');
    }
    $backUrl = '/crispvort/backend/uploads/kyc/' . $backFilename;

    // Check if KYC record already exists
    $stmt = $pdo->prepare("SELECT id FROM kyc_documents WHERE user_id = ?");
    $stmt->execute([$userId]);
    $existingKYC = $stmt->fetch();

    if ($existingKYC) {
        // Update existing KYC record
        $stmt = $pdo->prepare("
            UPDATE kyc_documents 
            SET id_type = ?, full_name = ?, address = ?, date_of_birth = ?, 
                front_id_url = ?, back_id_url = ?, status = 'pending', 
                updated_at = NOW()
            WHERE user_id = ?
        ");
        $stmt->execute([
            $idType, $fullName, $address, $dateOfBirth,
            $frontUrl, $backUrl, $userId
        ]);
    } else {
        // Insert new KYC record
        $stmt = $pdo->prepare("
            INSERT INTO kyc_documents 
            (user_id, id_type, full_name, address, date_of_birth, 
             front_id_url, back_id_url, status, created_at, updated_at)
            VALUES (?, ?, ?, ?, ?, ?, ?, 'pending', NOW(), NOW())
        ");
        $stmt->execute([
            $userId, $idType, $fullName, $address, $dateOfBirth,
            $frontUrl, $backUrl
        ]);
    }

    // Update user's KYC status
    $stmt = $pdo->prepare("UPDATE users SET kyc_status = 'pending' WHERE id = ?");
    $stmt->execute([$userId]);

    echo json_encode([
        'success' => true,
        'message' => 'KYC documents submitted successfully. Your documents are under review.',
    ]);

} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}
?>

