<?php
/**
 * P2P Trading Demo Data Setup Script
 * 
 * This script automatically sets up demo data for P2P trading.
 * It handles AUTO_INCREMENT fixes and inserts all demo data safely.
 * 
 * Usage: Run this file directly in your browser or via command line:
 * php setup_p2p_demo_data.php
 */

header("Access-Control-Allow-Origin: *");
header("Content-Type: application/json");

// Database configuration
$host = "localhost";
$dbname = "vortex";
$username = "root";
$password = "";

$response = [
    'success' => false,
    'message' => '',
    'steps' => [],
    'errors' => []
];

try {
    // Connect to database
    $pdo = new PDO("mysql:host=$host;dbname=$dbname;charset=utf8mb4", $username, $password);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
    $response['steps'][] = "✓ Connected to database";
    
    // Step 1: Check and fix AUTO_INCREMENT
    try {
        $stmt = $pdo->query("
            SELECT EXTRA 
            FROM INFORMATION_SCHEMA.COLUMNS 
            WHERE TABLE_SCHEMA = DATABASE() 
            AND TABLE_NAME = 'users' 
            AND COLUMN_NAME = 'id'
        ");
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($result && strpos($result['EXTRA'], 'auto_increment') === false) {
            $pdo->exec("ALTER TABLE users MODIFY COLUMN id INT AUTO_INCREMENT");
            $response['steps'][] = "✓ Fixed AUTO_INCREMENT on users.id";
        } else {
            $response['steps'][] = "✓ AUTO_INCREMENT already set on users.id";
        }
    } catch (PDOException $e) {
        $response['errors'][] = "AUTO_INCREMENT check failed: " . $e->getMessage();
    }
    
    // Step 2: Create demo users
    $demoUsers = [
        ['crypto_seller', 'seller@demo.com', 4.8, 5000],
        ['bitcoin_trader', 'trader@demo.com', 4.9, 8000],
        ['solana_expert', 'expert@demo.com', 4.7, 3000],
        ['eth_master', 'master@demo.com', 4.6, 6000],
        ['usdt_pro', 'pro@demo.com', 5.0, 10000],
    ];
    
    $userIds = [];
    $passwordHash = '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi'; // Demo password hash
    
    foreach ($demoUsers as $user) {
        try {
            // Check if user exists
            $stmt = $pdo->prepare("SELECT id FROM users WHERE username = ?");
            $stmt->execute([$user[0]]);
            $existing = $stmt->fetch();
            
            if ($existing) {
                $userIds[$user[0]] = $existing['id'];
                $response['steps'][] = "✓ User '{$user[0]}' already exists (ID: {$existing['id']})";
            } else {
                // Insert new user
                $stmt = $pdo->prepare("
                    INSERT INTO users (username, email, password, rating, balance, created_at) 
                    VALUES (?, ?, ?, ?, ?, NOW())
                ");
                $stmt->execute([$user[0], $user[1], $passwordHash, $user[2], $user[3]]);
                $userId = $pdo->lastInsertId();
                $userIds[$user[0]] = $userId;
                $response['steps'][] = "✓ Created user '{$user[0]}' (ID: $userId)";
            }
        } catch (PDOException $e) {
            $response['errors'][] = "Failed to create user '{$user[0]}': " . $e->getMessage();
        }
    }
    
    if (count($userIds) < 5) {
        throw new Exception("Failed to create all demo users. Only " . count($userIds) . " created.");
    }
    
    // Step 3: Clear existing demo listings (optional - comment out if you want to keep them)
    try {
        $pdo->exec("DELETE FROM listings WHERE seller_id IN (" . implode(',', $userIds) . ")");
        $response['steps'][] = "✓ Cleared existing demo listings";
    } catch (PDOException $e) {
        // Ignore if listings table doesn't exist yet
    }
    
    // Step 4: Insert demo listings
    $listings = [
        // Bitcoin
        [$userIds['crypto_seller'], 'BTC', 0.5, 0.01, 0.1, 45000.00, 'sell', 'PayPal', 'USD', 'USA', 'USD'],
        [$userIds['bitcoin_trader'], 'BTC', 1.2, 0.05, 0.5, 45200.00, 'sell', 'Bank Transfer', 'USD', 'USA', 'USD'],
        [$userIds['crypto_seller'], 'BTC', 0.3, 0.01, 0.15, 44800.00, 'sell', 'Wise', 'USD', 'UK', 'GBP'],
        [$userIds['bitcoin_trader'], 'BTC', 0.8, 0.02, 0.3, 45100.00, 'buy', 'PayPal', 'USD', 'USA', 'USD'],
        // Ethereum
        [$userIds['eth_master'], 'ETH', 5.0, 0.1, 2.0, 2800.00, 'sell', 'Bank Transfer', 'USD', 'USA', 'USD'],
        [$userIds['eth_master'], 'ETH', 3.5, 0.05, 1.5, 2820.00, 'sell', 'PayPal', 'USD', 'Canada', 'CAD'],
        [$userIds['crypto_seller'], 'ETH', 2.0, 0.1, 1.0, 2790.00, 'sell', 'Zelle', 'USD', 'USA', 'USD'],
        // Solana
        [$userIds['solana_expert'], 'SOL', 100, 5, 50, 95.50, 'sell', 'PayPal', 'USD', 'USA', 'USD'],
        [$userIds['solana_expert'], 'SOL', 200, 10, 100, 96.00, 'sell', 'Bank Transfer', 'USD', 'USA', 'USD'],
        [$userIds['bitcoin_trader'], 'SOL', 150, 5, 75, 94.80, 'sell', 'Wise', 'USD', 'UK', 'GBP'],
        [$userIds['solana_expert'], 'SOL', 80, 5, 40, 95.20, 'buy', 'PayPal', 'USD', 'USA', 'USD'],
        // USDT
        [$userIds['usdt_pro'], 'USDT', 10000, 100, 5000, 1.00, 'sell', 'Bank Transfer', 'USD', 'USA', 'USD'],
        [$userIds['usdt_pro'], 'USDT', 5000, 50, 2500, 0.99, 'sell', 'PayPal', 'USD', 'USA', 'USD'],
        [$userIds['crypto_seller'], 'USDT', 8000, 100, 4000, 1.01, 'sell', 'Wise', 'USD', 'UK', 'GBP'],
        [$userIds['usdt_pro'], 'USDT', 15000, 200, 7500, 1.00, 'buy', 'Bank Transfer', 'USD', 'USA', 'USD'],
    ];
    
    $listingIds = [];
    $stmt = $pdo->prepare("
        INSERT INTO listings (seller_id, coin, amount, min_limit, max_limit, price_per_unit, trade_type, payment_method, currency, country, accepted_currency, status, created_at) 
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'active', DATE_SUB(NOW(), INTERVAL ? MINUTE))
    ");
    
    $timeOffsets = [120, 60, 30, 15, 45, 20, 10, 60, 25, 5, 2, 40, 18, 8, 3];
    
    foreach ($listings as $index => $listing) {
        try {
            $timeOffset = $timeOffsets[$index] ?? 0;
            $stmt->execute(array_merge($listing, [$timeOffset]));
            $listingIds[] = $pdo->lastInsertId();
        } catch (PDOException $e) {
            $response['errors'][] = "Failed to create listing: " . $e->getMessage();
        }
    }
    
    $response['steps'][] = "✓ Created " . count($listingIds) . " demo listings";
    
    // Step 5: Create demo trades
    if (count($listingIds) >= 3) {
        $trades = [
            // Active trades
            [$listingIds[0], $userIds['bitcoin_trader'], $userIds['crypto_seller'], 0.05, 2250.00, 'USD', 'active', 20],
            [$listingIds[4], $userIds['crypto_seller'], $userIds['eth_master'], 0.5, 1400.00, 'USD', 'active', 15],
            [$listingIds[7], $userIds['bitcoin_trader'], $userIds['solana_expert'], 25, 2387.50, 'USD', 'active', 10],
            // Completed trades
            [$listingIds[0], $userIds['solana_expert'], $userIds['crypto_seller'], 0.1, 4500.00, 'USD', 'completed', 2880], // 2 days ago
            [$listingIds[4], $userIds['bitcoin_trader'], $userIds['eth_master'], 1.0, 2800.00, 'USD', 'completed', 1440], // 1 day ago
            [$listingIds[7], $userIds['crypto_seller'], $userIds['solana_expert'], 50, 4775.00, 'USD', 'completed', 4320], // 3 days ago
        ];
        
        $tradeIds = [];
        $stmtActive = $pdo->prepare("
            INSERT INTO trades (listing_id, buyer_id, seller_id, amount, total_price, currency, status, created_at) 
            VALUES (?, ?, ?, ?, ?, ?, ?, DATE_SUB(NOW(), INTERVAL ? MINUTE))
        ");
        
        $stmtCompleted = $pdo->prepare("
            INSERT INTO trades (listing_id, buyer_id, seller_id, amount, total_price, currency, status, created_at, completed_at) 
            VALUES (?, ?, ?, ?, ?, ?, ?, DATE_SUB(NOW(), INTERVAL ? MINUTE), DATE_SUB(NOW(), INTERVAL ? - 25 MINUTE))
        ");
        
        foreach ($trades as $trade) {
            try {
                $isCompleted = $trade[6] === 'completed';
                if ($isCompleted) {
                    $stmtCompleted->execute([
                        $trade[0], $trade[1], $trade[2], $trade[3], $trade[4], $trade[5], 
                        $trade[6], $trade[7], $trade[7]
                    ]);
                } else {
                    $stmtActive->execute([
                        $trade[0], $trade[1], $trade[2], $trade[3], $trade[4], $trade[5], 
                        $trade[6], $trade[7]
                    ]);
                }
                $tradeIds[] = $pdo->lastInsertId();
            } catch (PDOException $e) {
                $response['errors'][] = "Failed to create trade: " . $e->getMessage();
            }
        }
        
        $response['steps'][] = "✓ Created " . count($tradeIds) . " demo trades";
        
        // Step 6: Create chat messages
        if (count($tradeIds) >= 3) {
            $messages = [
                [$tradeIds[0], $userIds['crypto_seller'], $userIds['bitcoin_trader'], 'crypto_seller', 'bitcoin_trader', "Hello! I'm ready to proceed with the trade.", 18],
                [$tradeIds[0], $userIds['bitcoin_trader'], $userIds['crypto_seller'], 'bitcoin_trader', 'crypto_seller', "Great! I've sent the payment via PayPal.", 15],
                [$tradeIds[0], $userIds['crypto_seller'], $userIds['bitcoin_trader'], 'crypto_seller', 'bitcoin_trader', "Payment received! I'll release the coins now.", 12],
                [$tradeIds[1], $userIds['eth_master'], $userIds['crypto_seller'], 'eth_master', 'crypto_seller', "Hi, I'm interested in buying 0.5 ETH.", 13],
                [$tradeIds[1], $userIds['crypto_seller'], $userIds['eth_master'], 'crypto_seller', 'eth_master', "Perfect! Please send payment and I'll confirm.", 10],
                [$tradeIds[3], $userIds['solana_expert'], $userIds['crypto_seller'], 'solana_expert', 'crypto_seller', "Thanks for the smooth trade!", 2880 + 20],
                [$tradeIds[3], $userIds['crypto_seller'], $userIds['solana_expert'], 'crypto_seller', 'solana_expert', "You're welcome! Great doing business with you.", 2880 + 22],
            ];
            
            $stmt = $pdo->prepare("
                INSERT INTO chat (trade_id, sender_id, receiver_id, sender_username, receiver_username, message, created_at) 
                VALUES (?, ?, ?, ?, ?, ?, DATE_SUB(NOW(), INTERVAL ? MINUTE))
            ");
            
            $messageCount = 0;
            foreach ($messages as $msg) {
                try {
                    $stmt->execute($msg);
                    $messageCount++;
                } catch (PDOException $e) {
                    $response['errors'][] = "Failed to create chat message: " . $e->getMessage();
                }
            }
            
            $response['steps'][] = "✓ Created $messageCount demo chat messages";
            
            // Step 7: Create ratings
            if (count($tradeIds) >= 3) {
                $ratings = [
                    [$tradeIds[3], $userIds['crypto_seller'], $userIds['solana_expert'], 5, "Excellent trader, very professional!"],
                    [$tradeIds[3], $userIds['solana_expert'], $userIds['crypto_seller'], 5, "Fast and reliable, highly recommended!"],
                ];
                
                $stmt = $pdo->prepare("
                    INSERT INTO ratings (trade_id, rater_id, rated_user_id, rating, comment, created_at) 
                    VALUES (?, ?, ?, ?, ?, DATE_SUB(NOW(), INTERVAL 2880 + 25 MINUTE))
                ");
                
                $ratingCount = 0;
                foreach ($ratings as $rating) {
                    try {
                        $stmt->execute($rating);
                        $ratingCount++;
                    } catch (PDOException $e) {
                        $response['errors'][] = "Failed to create rating: " . $e->getMessage();
                    }
                }
                
                $response['steps'][] = "✓ Created $ratingCount demo ratings";
                
                // Update user ratings
                try {
                    $pdo->exec("
                        UPDATE users u 
                        SET rating = (
                            SELECT ROUND(AVG(rating), 1) 
                            FROM ratings r 
                            WHERE r.rated_user_id = u.id
                        )
                        WHERE u.id IN (SELECT DISTINCT rated_user_id FROM ratings)
                    ");
                    $response['steps'][] = "✓ Updated user ratings";
                } catch (PDOException $e) {
                    $response['errors'][] = "Failed to update ratings: " . $e->getMessage();
                }
            }
        }
    }
    
    $response['success'] = true;
    $response['message'] = "Demo data setup completed successfully!";
    $response['summary'] = [
        'users' => count($userIds),
        'listings' => count($listingIds),
        'trades' => isset($tradeIds) ? count($tradeIds) : 0,
    ];
    
} catch (PDOException $e) {
    $response['success'] = false;
    $response['message'] = "Database error: " . $e->getMessage();
    $response['errors'][] = $e->getMessage();
} catch (Exception $e) {
    $response['success'] = false;
    $response['message'] = "Error: " . $e->getMessage();
    $response['errors'][] = $e->getMessage();
}

// Output response
echo json_encode($response, JSON_PRETTY_PRINT);

// If running from command line, also print to console
if (php_sapi_name() === 'cli') {
    echo "\n\n";
    echo "=== Setup Summary ===\n";
    echo "Success: " . ($response['success'] ? 'Yes' : 'No') . "\n";
    echo "Message: " . $response['message'] . "\n";
    if (isset($response['summary'])) {
        echo "Users created: " . $response['summary']['users'] . "\n";
        echo "Listings created: " . $response['summary']['listings'] . "\n";
        echo "Trades created: " . $response['summary']['trades'] . "\n";
    }
    if (!empty($response['errors'])) {
        echo "\nErrors:\n";
        foreach ($response['errors'] as $error) {
            echo "  - $error\n";
        }
    }
    echo "\nSteps completed:\n";
    foreach ($response['steps'] as $step) {
        echo "  $step\n";
    }
}
?>

