<?php
header("Access-Control-Allow-Origin: *");
header("Content-Type: application/json; charset=UTF-8");

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Error logging function
function logError($message, $data = null) {
    $logMessage = date("Y-m-d H:i:s") . " - ERROR: " . $message;
    if ($data !== null) {
        $logMessage .= " | Data: " . json_encode($data);
    }
    $logMessage .= PHP_EOL;
    file_put_contents('debug.log', $logMessage, FILE_APPEND);
    error_log($logMessage);
}

// Success logging function
function logInfo($message, $data = null) {
    $logMessage = date("Y-m-d H:i:s") . " - INFO: " . $message;
    if ($data !== null) {
        $logMessage .= " | Data: " . json_encode($data);
    }
    $logMessage .= PHP_EOL;
    file_put_contents('debug.log', $logMessage, FILE_APPEND);
}

// Initialize response
$response = ["success" => false, "error" => ""];

// Function to ensure status ENUM includes 'pending'
function ensureStatusEnum($conn) {
    // Check current ENUM values
    $result = $conn->query("SHOW COLUMNS FROM trades WHERE Field = 'status'");
    if ($result && $row = $result->fetch_assoc()) {
        $type = $row['Type'];
        // Check if 'pending' is already in the ENUM
        if (strpos($type, "'pending'") === false) {
            logInfo("Updating status ENUM to include 'pending' and 'waiting_seller'");
            // Update the ENUM to include pending and waiting_seller
            $alterQuery = "ALTER TABLE trades 
                MODIFY COLUMN status ENUM('pending', 'waiting_seller', 'active', 'completed', 'cancelled', 'disputed') DEFAULT 'pending'";
            
            if (!$conn->query($alterQuery)) {
                logError("Failed to update status ENUM: " . $conn->error);
                return false;
            }
            logInfo("Status ENUM updated successfully");
        }
    }
    return true;
}

try {
    // Log incoming request
    $rawInput = file_get_contents("php://input");
    logInfo("Incoming request", ["raw_length" => strlen($rawInput)]);
    
    // Database connection
    $conn = new mysqli("localhost","coinfzbe_pro_user", "_xS89yo#g?Rv", "coinfzbe_vortexprp");
    if ($conn->connect_error) {
        $errorMsg = "Database connection failed: " . $conn->connect_error;
        logError($errorMsg);
        $response["error"] = "Database connection failed. Please try again later.";
        echo json_encode($response);
        exit;
    }
    logInfo("Database connected successfully");
    
    // Ensure status ENUM includes 'pending'
    if (!ensureStatusEnum($conn)) {
        $response["error"] = "Database schema update failed. Please contact support.";
        echo json_encode($response);
        exit;
    }

    // Get and decode POST body
    $data = json_decode($rawInput, true);
    
    if (json_last_error() !== JSON_ERROR_NONE) {
        $errorMsg = "Invalid JSON: " . json_last_error_msg();
        logError($errorMsg, ["json_input" => substr($rawInput, 0, 500)]);
        $response["error"] = "Invalid request data. Please refresh and try again.";
        echo json_encode($response);
        exit;
    }
    
    logInfo("JSON decoded successfully", $data);

    // Validate required fields
    $missingFields = [];
    if (!isset($data['listing_id']) || empty($data['listing_id'])) {
        $missingFields[] = "listing_id";
    }
    if (!isset($data['buyer_id']) || empty($data['buyer_id'])) {
        $missingFields[] = "buyer_id";
    }
    if (!isset($data['amount']) || empty($data['amount'])) {
        $missingFields[] = "amount";
    }
    
    if (!empty($missingFields)) {
        $errorMsg = "Missing required fields: " . implode(", ", $missingFields);
        logError($errorMsg, $data);
        $response["error"] = "Missing required information: " . implode(", ", $missingFields) . ". Please refresh and try again.";
        echo json_encode($response);
        exit;
    }

    $listing_id = intval($data['listing_id']);
    $buyer_id = intval($data['buyer_id']);
    $amount = floatval($data['amount']);

    logInfo("Validated input", [
        "listing_id" => $listing_id,
        "buyer_id" => $buyer_id,
        "amount" => $amount
    ]);

    // Validate amount is positive
    if ($amount <= 0) {
        $errorMsg = "Invalid amount: amount must be greater than 0";
        logError($errorMsg, ["amount" => $amount]);
        $response["error"] = "Trade amount must be greater than 0.";
        echo json_encode($response);
        exit;
    }

    // Get listing to fetch seller, price
    $stmt = $conn->prepare("SELECT seller_id, price_per_unit, coin, status FROM listings WHERE id = ?");
    if (!$stmt) {
        $errorMsg = "Failed to prepare statement: " . $conn->error;
        logError($errorMsg);
        $response["error"] = "Database error occurred. Please try again.";
        echo json_encode($response);
        exit;
    }

    $stmt->bind_param("i", $listing_id);
    if (!$stmt->execute()) {
        $errorMsg = "Failed to execute listing query: " . $stmt->error;
        logError($errorMsg, ["listing_id" => $listing_id]);
        $response["error"] = "Failed to retrieve listing information. Please try again.";
        echo json_encode($response);
        exit;
    }

    $result = $stmt->get_result();
    if ($result->num_rows === 0) {
        $errorMsg = "Listing not found";
        logError($errorMsg, ["listing_id" => $listing_id]);
        $response["error"] = "Listing not found. It may have been removed or is no longer available.";
        echo json_encode($response);
        exit;
    }

    $listing = $result->fetch_assoc();
    $seller_id = $listing['seller_id'];
    $price_per_unit = $listing['price_per_unit'];
    $listing_status = $listing['status'] ?? 'active';

    // Check if listing is active
    if ($listing_status !== 'active') {
        $errorMsg = "Listing is not active";
        logError($errorMsg, ["listing_id" => $listing_id, "status" => $listing_status]);
        $response["error"] = "This listing is no longer available for trading (status: " . $listing_status . ").";
        echo json_encode($response);
        exit;
    }

    // Check if buyer is trying to trade with themselves
    if ($buyer_id == $seller_id) {
        $errorMsg = "Buyer cannot trade with themselves";
        logError($errorMsg, ["buyer_id" => $buyer_id, "seller_id" => $seller_id]);
        $response["error"] = "You cannot create a trade with your own listing.";
        echo json_encode($response);
        exit;
    }

    logInfo("Listing retrieved", [
        "seller_id" => $seller_id,
        "price_per_unit" => $price_per_unit,
        "listing_status" => $listing_status
    ]);

    // Calculate total price
    $total_price = $amount * $price_per_unit;

    if ($total_price <= 0) {
        $errorMsg = "Invalid total price calculation";
        logError($errorMsg, [
            "amount" => $amount,
            "price_per_unit" => $price_per_unit,
            "total_price" => $total_price
        ]);
        $response["error"] = "Invalid trade calculation. Please refresh and try again.";
        echo json_encode($response);
        exit;
    }

    // Insert trade with pending status - seller needs to accept and fund escrow
    $stmt = $conn->prepare("INSERT INTO trades (listing_id, buyer_id, seller_id, amount, total_price, status, created_at)
VALUES (?, ?, ?, ?, ?, 'pending', NOW())");
    
    if (!$stmt) {
        $errorMsg = "Failed to prepare insert statement: " . $conn->error;
        logError($errorMsg);
        $response["error"] = "Database error occurred while creating trade. Please try again.";
        echo json_encode($response);
        exit;
    }

    $stmt->bind_param("iiidd", $listing_id, $buyer_id, $seller_id, $amount, $total_price);
    
    if (!$stmt->execute()) {
        $errorMsg = "Failed to insert trade: " . $stmt->error;
        $errorCode = $stmt->errno;
        logError($errorMsg, [
            "error_code" => $errorCode,
            "listing_id" => $listing_id,
            "buyer_id" => $buyer_id,
            "seller_id" => $seller_id,
            "amount" => $amount,
            "total_price" => $total_price
        ]);
        
        // Provide more specific error messages based on error code
        if ($errorCode == 1062) {
            $response["error"] = "A trade for this listing already exists. Please check your active trades.";
        } elseif ($errorCode == 1452) {
            $response["error"] = "Invalid listing or user ID. Please refresh and try again.";
        } elseif (strpos($stmt->error, 'Data truncated') !== false || strpos($stmt->error, 'status') !== false) {
            // Status ENUM issue - try to fix it
            logError("Status ENUM issue detected, attempting to fix schema");
            if (ensureStatusEnum($conn)) {
                // Retry the insert
                $retryStmt = $conn->prepare("INSERT INTO trades (listing_id, buyer_id, seller_id, amount, total_price, status, created_at)
VALUES (?, ?, ?, ?, ?, 'pending', NOW())");
                $retryStmt->bind_param("iiidd", $listing_id, $buyer_id, $seller_id, $amount, $total_price);
                
                if ($retryStmt->execute()) {
                    $trade_id = $conn->insert_id;
                    logInfo("Trade created successfully after schema fix", ["trade_id" => $trade_id]);
                    $response["success"] = true;
                    $response["trade_id"] = $trade_id;
                    echo json_encode($response);
                    exit;
                } else {
                    $response["error"] = "Failed to create trade after schema update. Error: " . $retryStmt->error;
                }
            } else {
                $response["error"] = "Database schema issue: The 'pending' status is not available. Please contact support to update the database schema.";
            }
        } else {
            $response["error"] = "Failed to create trade: " . $stmt->error . " (Error code: " . $errorCode . ")";
        }
        
        echo json_encode($response);
        exit;
    }

    $trade_id = $conn->insert_id;
    logInfo("Trade created successfully", [
        "trade_id" => $trade_id,
        "listing_id" => $listing_id,
        "buyer_id" => $buyer_id,
        "amount" => $amount
    ]);

    $response["success"] = true;
    $response["trade_id"] = $trade_id;
    echo json_encode($response);

} catch (Exception $e) {
    $errorMsg = "Unexpected error: " . $e->getMessage();
    logError($errorMsg, [
        "file" => $e->getFile(),
        "line" => $e->getLine(),
        "trace" => $e->getTraceAsString()
    ]);
    $response["error"] = "An unexpected error occurred: " . $e->getMessage();
    echo json_encode($response);
} finally {
    if (isset($conn)) {
        $conn->close();
    }
}
?>
