<?php
// cancel_trade.php
// Allows seller to cancel a trade before escrow is funded
header("Access-Control-Allow-Origin: *");
header("Content-Type: application/json");

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/notification_helper.php';

try {
    $data = json_decode(file_get_contents("php://input"), true);
    $tradeId = $data['trade_id'] ?? 0;
    $userId = $data['user_id'] ?? 0;
    $reason = $data['reason'] ?? 'No reason provided';
    
    if (!$tradeId || !$userId) {
        http_response_code(400);
        echo json_encode(['success' => false, 'error' => 'Trade ID and User ID are required']);
        exit;
    }
    
    // Get trade details
    $stmt = $pdo->prepare("SELECT * FROM trades WHERE id = ?");
    $stmt->execute([$tradeId]);
    $trade = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$trade) {
        http_response_code(404);
        echo json_encode(['success' => false, 'error' => 'Trade not found']);
        exit;
    }
    
    // Check if user is the seller
    if ($userId != $trade['seller_id']) {
        http_response_code(403);
        echo json_encode(['success' => false, 'error' => 'Only the seller can cancel this trade']);
        exit;
    }
    
    // Check if trade can be cancelled (only if pending or waiting_seller, and escrow not funded)
    if ($trade['status'] !== 'pending' && $trade['status'] !== 'waiting_seller') {
        http_response_code(400);
        echo json_encode([
            'success' => false, 
            'error' => 'Trade cannot be cancelled. It is already ' . $trade['status']
        ]);
        exit;
    }
    
    // Check if escrow is funded (if seller_payment_details exists, escrow might be funded)
    // For now, we'll allow cancellation if status is pending or waiting_seller
    // In a real system, you'd check if escrow balance > 0
    
    // Update trade status
    $updateStmt = $pdo->prepare("
        UPDATE trades 
        SET status = 'cancelled',
            cancel_reason = ?,
            cancelled_by = ?,
            cancelled_at = NOW()
        WHERE id = ?
    ");
    $updateStmt->execute([$reason, $userId, $tradeId]);
    
    // Add system message to chat
    $msgStmt = $pdo->prepare("
        INSERT INTO chat (trade_id, sender_id, receiver_id, message, created_at)
        VALUES (?, ?, ?, ?, NOW())
    ");
    
    $buyerId = $trade['buyer_id'];
    $userStmt = $pdo->prepare("SELECT username FROM users WHERE id = ?");
    $userStmt->execute([$userId]);
    $userData = $userStmt->fetch(PDO::FETCH_ASSOC);
    $username = $userData['username'] ?? 'Seller';
    
    $msgStmt->execute([
        $tradeId,
        $userId,
        $buyerId,
        "❌ Trade cancelled by seller. Reason: {$reason}"
    ]);
    
    // Notify buyer
    sendNotification($buyerId, 'Trade Cancelled', 'The seller has cancelled this trade.', [
        'type' => 'trade_cancelled',
        'trade_id' => $tradeId,
        'reason' => $reason
    ]);
    
    echo json_encode([
        'success' => true,
        'message' => 'Trade cancelled successfully',
        'cancelled' => true
    ]);
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => 'Database error: ' . $e->getMessage()]);
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => $e->getMessage()]);
}
?>
