<?php
// add_admin_to_trade.php
// Allows users to request admin assistance or admin to join a trade chat
header("Access-Control-Allow-Origin: *");
header("Content-Type: application/json");

// Enable error logging
ini_set('display_errors', 0);
ini_set('log_errors', 1);
error_reporting(E_ALL);

// Log function
function logError($message, $data = null) {
    $logMessage = date("Y-m-d H:i:s") . " - [add_admin_to_trade] " . $message;
    if ($data !== null) {
        $logMessage .= " | Data: " . json_encode($data);
    }
    $logMessage .= PHP_EOL;
    error_log($logMessage);
    file_put_contents(__DIR__ . '/admin_trade_errors.log', $logMessage, FILE_APPEND);
}

logError("Request received", $_SERVER['REQUEST_METHOD']);

require_once __DIR__ . '/../config/database.php';

// Check if notification_helper exists
$notificationHelperPath = __DIR__ . '/notification_helper.php';
if (!file_exists($notificationHelperPath)) {
    logError("Notification helper not found", ['path' => $notificationHelperPath]);
    // Continue without notification helper
    function sendNotification($userId, $title, $message, $data = []) {
        logError("sendNotification called but helper not loaded", ['userId' => $userId, 'title' => $title]);
        return false;
    }
} else {
    require_once $notificationHelperPath;
}

try {
    $rawInput = file_get_contents("php://input");
    logError("Raw input received", ['length' => strlen($rawInput), 'content' => substr($rawInput, 0, 200)]);
    
    $data = json_decode($rawInput, true);
    
    if (json_last_error() !== JSON_ERROR_NONE) {
        logError("JSON decode error", ['error' => json_last_error_msg(), 'input' => $rawInput]);
        http_response_code(400);
        echo json_encode(['success' => false, 'error' => 'Invalid JSON: ' . json_last_error_msg()]);
        exit;
    }
    
    logError("JSON decoded successfully", $data);
    
    $tradeId = $data['trade_id'] ?? 0;
    $userId = $data['user_id'] ?? 0;
    $adminId = $data['admin_id'] ?? null; // If provided, admin is joining directly
    
    logError("Parsed parameters", ['tradeId' => $tradeId, 'userId' => $userId, 'adminId' => $adminId]);
    
    if (!$tradeId || !$userId) {
        logError("Missing required parameters", ['tradeId' => $tradeId, 'userId' => $userId]);
        http_response_code(400);
        echo json_encode(['success' => false, 'error' => 'Trade ID and User ID are required']);
        exit;
    }
    
    // Check database connection
    if (!isset($pdo)) {
        logError("PDO not initialized");
        http_response_code(500);
        echo json_encode(['success' => false, 'error' => 'Database connection not initialized']);
        exit;
    }
    
    // Get trade details
    try {
        logError("Fetching trade details", ['tradeId' => $tradeId]);
        $stmt = $pdo->prepare("SELECT * FROM trades WHERE id = ?");
        $stmt->execute([$tradeId]);
        $trade = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$trade) {
            logError("Trade not found", ['tradeId' => $tradeId]);
            http_response_code(404);
            echo json_encode(['success' => false, 'error' => 'Trade not found']);
            exit;
        }
        
        logError("Trade found", ['tradeId' => $tradeId, 'buyerId' => $trade['buyer_id'] ?? 'N/A', 'sellerId' => $trade['seller_id'] ?? 'N/A']);
    } catch (PDOException $e) {
        logError("Error fetching trade", ['error' => $e->getMessage(), 'code' => $e->getCode()]);
        http_response_code(500);
        echo json_encode(['success' => false, 'error' => 'Database error fetching trade: ' . $e->getMessage()]);
        exit;
    }
    
    // Check if user is admin
    try {
        logError("Checking user role", ['userId' => $userId]);
        $userStmt = $pdo->prepare("SELECT role FROM users WHERE id = ?");
        $userStmt->execute([$userId]);
        $user = $userStmt->fetch(PDO::FETCH_ASSOC);
        $isAdmin = $user && ($user['role'] === 'admin' || $user['role'] === 'Admin');
        logError("User role check", ['userId' => $userId, 'role' => $user['role'] ?? 'N/A', 'isAdmin' => $isAdmin]);
    } catch (PDOException $e) {
        logError("Error checking user role", ['error' => $e->getMessage()]);
        http_response_code(500);
        echo json_encode(['success' => false, 'error' => 'Database error checking user: ' . $e->getMessage()]);
        exit;
    }
    
    if ($isAdmin && $adminId) {
        // Admin is joining directly
        // Try to update with admin columns, catch error if they don't exist
        try {
            $updateStmt = $pdo->prepare("
                UPDATE trades 
                SET admin_id = ?, 
                    admin_joined_at = NOW() 
                WHERE id = ?
            ");
            $updateStmt->execute([$adminId, $tradeId]);
        } catch (PDOException $e) {
            // If columns don't exist, throw a helpful error
            http_response_code(500);
            echo json_encode([
                'success' => false, 
                'error' => 'Database migration required. Please run migrate_trades_admin_dispute.sql first.',
                'details' => $e->getMessage()
            ]);
            exit;
        }
        
        // Send notification to both parties
        $buyerId = $trade['buyer_id'];
        $sellerId = $trade['seller_id'];
        
        sendNotification($buyerId, 'Admin Joined', 'An admin has joined your trade chat to assist.', [
            'type' => 'admin_joined',
            'trade_id' => $tradeId
        ]);
        
        sendNotification($sellerId, 'Admin Joined', 'An admin has joined your trade chat to assist.', [
            'type' => 'admin_joined',
            'trade_id' => $tradeId
        ]);
        
        // Add system message to chat
        $msgStmt = $pdo->prepare("
            INSERT INTO chat (trade_id, sender_id, receiver_id, message, created_at)
            VALUES (?, ?, ?, ?, NOW())
        ");
        $adminUserStmt = $pdo->prepare("SELECT username FROM users WHERE id = ?");
        $adminUserStmt->execute([$adminId]);
        $adminUser = $adminUserStmt->fetch(PDO::FETCH_ASSOC);
        $adminUsername = $adminUser['username'] ?? 'Admin';
        
        // Message to buyer
        $msgStmt->execute([
            $tradeId,
            $adminId,
            $buyerId,
            "🔧 Admin {$adminUsername} has joined this chat to assist with your trade."
        ]);
        
        // Message to seller
        $msgStmt->execute([
            $tradeId,
            $adminId,
            $sellerId,
            "🔧 Admin {$adminUsername} has joined this chat to assist with your trade."
        ]);
        
        echo json_encode([
            'success' => true,
            'message' => 'Admin joined successfully',
            'admin_id' => $adminId
        ]);
    } else {
        // Regular user requesting admin assistance
        // Check if admin is already in chat (handle case where column might not exist)
        if (isset($trade['admin_id']) && $trade['admin_id']) {
            echo json_encode([
                'success' => true,
                'message' => 'Admin is already in this chat',
                'admin_id' => $trade['admin_id']
            ]);
            exit;
        }
        
        // Add system message requesting admin
        try {
            logError("Adding system message to chat");
            $otherPartyId = $userId == $trade['buyer_id'] ? $trade['seller_id'] : $trade['buyer_id'];
            logError("Other party ID", ['otherPartyId' => $otherPartyId]);
            
            $userStmt = $pdo->prepare("SELECT username FROM users WHERE id = ?");
            $userStmt->execute([$userId]);
            $userData = $userStmt->fetch(PDO::FETCH_ASSOC);
            $username = $userData['username'] ?? 'User';
            logError("User data fetched", ['username' => $username]);
            
            $msgStmt = $pdo->prepare("
                INSERT INTO chat (trade_id, sender_id, receiver_id, message, created_at)
                VALUES (?, ?, ?, ?, NOW())
            ");
            $message = "🆘 {$username} has requested admin assistance for this trade.";
            logError("Executing chat insert", ['tradeId' => $tradeId, 'senderId' => $userId, 'receiverId' => $otherPartyId]);
            
            $msgStmt->execute([
                $tradeId,
                $userId,
                $otherPartyId,
                $message
            ]);
            
            logError("Chat message inserted successfully");
            
        } catch (PDOException $e) {
            logError("Error inserting chat message", ['error' => $e->getMessage(), 'code' => $e->getCode()]);
            http_response_code(500);
            echo json_encode([
                'success' => false, 
                'error' => 'Failed to add message to chat: ' . $e->getMessage(),
                'details' => $e->getMessage()
            ]);
            exit;
        }
        
        // TODO: Notify all admins about the request
        // For now, just return success
        logError("Request completed successfully");
        echo json_encode([
            'success' => true,
            'message' => 'Admin assistance requested. An admin will join shortly.',
            'admin_requested' => true
        ]);
    }
} catch (PDOException $e) {
    logError("PDO Exception caught", ['error' => $e->getMessage(), 'code' => $e->getCode(), 'file' => $e->getFile(), 'line' => $e->getLine()]);
    http_response_code(500);
    echo json_encode([
        'success' => false, 
        'error' => 'Database error: ' . $e->getMessage(),
        'details' => $e->getMessage()
    ]);
} catch (Exception $e) {
    logError("General Exception caught", ['error' => $e->getMessage(), 'file' => $e->getFile(), 'line' => $e->getLine()]);
    http_response_code(500);
    echo json_encode([
        'success' => false, 
        'error' => $e->getMessage(),
        'details' => $e->getMessage()
    ]);
}
?>

