<?php
header("Access-Control-Allow-Origin: *");
header("Content-Type: application/json; charset=UTF-8");

$conn = new mysqli("localhost","coinfzbe_pro_user", "_xS89yo#g?Rv", "coinfzbe_vortexprp");
if ($conn->connect_error) {
    echo json_encode(["success" => false, "error" => "DB connection failed"]);
    exit;
}

// Connection for transactions table (vortex database)
$transactions_conn = new mysqli("localhost","coinfzbe_pro_user", "_xS89yo#g?Rv", "coinfzbe_vortexprp");
if ($transactions_conn->connect_error) {
    // Log error but don't fail - transactions saving is secondary
    error_log("Transactions DB connection failed: " . $transactions_conn->connect_error);
}

$data = json_decode(file_get_contents("php://input"), true);

if (
    !isset($data['trade_id']) ||
    !isset($data['seller_id']) ||
    !isset($data['payment_details'])
) {
    echo json_encode(["success" => false, "error" => "Missing required fields"]);
    exit;
}

$trade_id = intval($data['trade_id']);
$seller_id = intval($data['seller_id']);
$payment_details = $conn->real_escape_string($data['payment_details']);

// Get trade with listing details
$stmt = $conn->prepare("
    SELECT t.*, l.coin, l.seller_id as listing_seller_id
    FROM trades t
    JOIN listings l ON t.listing_id = l.id
    WHERE t.id = ? AND l.seller_id = ?
");
$stmt->bind_param("ii", $trade_id, $seller_id);
$stmt->execute();
$result = $stmt->get_result();

if ($result->num_rows === 0) {
    echo json_encode(["success" => false, "error" => "Trade not found or unauthorized"]);
    exit;
}

$trade = $result->fetch_assoc();

// Check if trade is already accepted
if ($trade['status'] !== 'pending' && $trade['status'] !== 'waiting_seller') {
    echo json_encode(["success" => false, "error" => "Trade is not in pending state"]);
    exit;
}

$coin = strtolower($trade['coin']);
$amount = floatval($trade['amount']);

// Whitelist of valid coin balance columns for security
$valid_coins = ['btc', 'eth', 'usdt', 'usdc', 'sol', 'bnb', 'ada', 'xrp', 'doge', 'dot', 'matic', 'ltc', 'bch', 'xlm', 'link', 'uni', 'avax', 'atom', 'etc', 'trx'];
if (!in_array($coin, $valid_coins)) {
    echo json_encode([
        "success" => false,
        "error" => "Unsupported coin: {$trade['coin']}"
    ]);
    exit;
}

// Get seller's balance for the coin - use direct query with whitelisted column
$balance_key = "v_{$coin}";
$balance_query = "SELECT `{$balance_key}` FROM users WHERE id = ?";
$stmt = $conn->prepare($balance_query);
if (!$stmt) {
    error_log("Prepare failed for balance check: " . $conn->error);
    echo json_encode([
        "success" => false,
        "error" => "Database error while checking balance"
    ]);
    exit;
}

$stmt->bind_param("i", $seller_id);
$stmt->execute();
$balance_result = $stmt->get_result();
$seller_data = $balance_result->fetch_assoc();

// Check if seller data exists and get balance
if (!$seller_data) {
    error_log("Seller not found: seller_id={$seller_id}");
    echo json_encode([
        "success" => false,
        "error" => "Seller account not found"
    ]);
    exit;
}

$seller_balance = floatval($seller_data[$balance_key] ?? 0);

// Log balance check for debugging
error_log("Balance check - Seller ID: {$seller_id}, Coin: {$coin}, Balance Key: {$balance_key}, Current Balance: {$seller_balance}, Required: {$amount}");

if ($seller_balance < $amount) {
    $needed = $amount - $seller_balance;
    echo json_encode([
        "success" => false,
        "error" => "Insufficient balance. You have {$seller_balance} {$trade['coin']} but need {$amount} {$trade['coin']}. You need {$needed} more {$trade['coin']}."
    ]);
    exit;
}

// Start transaction
$conn->begin_transaction();

try {
    // Get seller username for transaction record
    $stmt = $conn->prepare("SELECT username FROM users WHERE id = ?");
    if (!$stmt) {
        throw new Exception("Failed to prepare seller username query: " . $conn->error);
    }
    $stmt->bind_param("i", $seller_id);
    if (!$stmt->execute()) {
        throw new Exception("Failed to get seller username: " . $stmt->error);
    }
    $seller_result = $stmt->get_result();
    $seller_user = $seller_result->fetch_assoc();
    $seller_username = $seller_user['username'] ?? '';
    
    if (empty($seller_username)) {
        error_log("Warning: Seller username is empty for seller_id: {$seller_id}");
    }

    // Generate transaction ID
    function generateRandomString($length = 17) {
        $characters = '0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ';
        $charactersLength = strlen($characters);
        $randomString = '';
        for ($i = 0; $i < $length; $i++) {
            $randomString .= 'URI'.$characters[random_int(0, $charactersLength - 1)];
        }
        return $randomString;
    }
    $txnid = generateRandomString(13);

    // Deduct from seller's balance (fund escrow)
    // Note: Using direct query for dynamic column names (coin is whitelisted, amount is validated)
    $amount_escaped = $conn->real_escape_string($amount);
    $seller_id_escaped = intval($seller_id); // Already validated as integer
    $update_query = "UPDATE users SET `{$balance_key}` = `{$balance_key}` - {$amount_escaped} WHERE id = {$seller_id_escaped}";
    
    if (!$conn->query($update_query)) {
        throw new Exception("Failed to update balance: " . $conn->error);
    }
    
    // Verify the update was successful
    $verify_query = "SELECT `{$balance_key}` FROM users WHERE id = {$seller_id_escaped}";
    $verify_result = $conn->query($verify_query);
    if ($verify_result) {
        $verify_data = $verify_result->fetch_assoc();
        $new_balance = floatval($verify_data[$balance_key] ?? 0);
        error_log("Balance updated - Seller ID: {$seller_id_escaped}, Coin: {$coin}, New Balance: {$new_balance}");
    }

    // Save transaction to transactions table - Seller escrow funding
    if ($transactions_conn && !$transactions_conn->connect_error) {
        $coin_upper = strtoupper($trade['coin']);
        $stmt_tx = $transactions_conn->prepare("
            INSERT INTO transactions (username, address, destination, amount, to_amount, status, type, coin, cointicker, txtype, txnid, fee)
            VALUES (?, 'p2p', 'escrow', ?, ?, 'Completed', 'Send', ?, ?, 'p2p', ?, '0.00000')
        ");
        if ($stmt_tx) {
            // Parameters: username(s), amount(d), to_amount(d), coin(s), cointicker(s), txnid(s) = 6 params
            // Format string: sddsss = 6 characters
            $stmt_tx->bind_param("sddsss", $seller_username, $amount, $amount, $trade['coin'], $coin_upper, $txnid);
            if (!$stmt_tx->execute()) {
                error_log("Warning: Failed to save transaction record: " . $stmt_tx->error);
            }
            $stmt_tx->close();
        }
    }

    // Update trade status and add payment details
    $stmt = $conn->prepare("
        UPDATE trades 
        SET status = 'active', 
            payment_details = ?,
            accepted_at = NOW()
        WHERE id = ?
    ");
    if (!$stmt) {
        throw new Exception("Failed to prepare trade update: " . $conn->error);
    }
    $stmt->bind_param("si", $payment_details, $trade_id);
    if (!$stmt->execute()) {
        throw new Exception("Failed to update trade: " . $stmt->error);
    }

    // Get buyer info for notifications
    $buyer_id = intval($trade['buyer_id']);
    $stmt = $conn->prepare("SELECT username FROM users WHERE id = ?");
    if (!$stmt) {
        throw new Exception("Failed to prepare buyer query: " . $conn->error);
    }
    $stmt->bind_param("i", $buyer_id);
    if (!$stmt->execute()) {
        throw new Exception("Failed to get buyer info: " . $stmt->error);
    }
    $buyer_result = $stmt->get_result();
    $buyer_user = $buyer_result->fetch_assoc();
    $buyer_username = $buyer_user['username'] ?? '';

    $conn->commit();
    
    // Send real-time notifications (wrap in try-catch so they don't fail the request)
    try {
        $notification_file = __DIR__ . '/../crispapi/notification_helper.php';
        if (file_exists($notification_file)) {
            require_once $notification_file;
            
            // Notify buyer
            if (function_exists('sendNotification') && $buyer_id && $buyer_username) {
                sendNotification($buyer_id, $buyer_username, [
                    'type' => 'trade_update',
                    'title' => 'Trade Accepted',
                    'message' => "Seller has accepted and funded the trade",
                    'data' => [
                        'tradeId' => $trade_id,
                        'status' => 'active',
                        'type' => 'p2p',
                    ],
                    'timestamp' => date('c'),
                ]);
            }
            
            // Notify seller
            if (function_exists('sendNotification') && $seller_id && $seller_username) {
                sendNotification($seller_id, $seller_username, [
                    'type' => 'trade_update',
                    'title' => 'Trade Accepted',
                    'message' => "You have accepted and funded the trade",
                    'data' => [
                        'tradeId' => $trade_id,
                        'status' => 'active',
                        'type' => 'p2p',
                    ],
                    'timestamp' => date('c'),
                ]);
            }
            
            // Send trade message update
            if (function_exists('sendTradeNotification')) {
                sendTradeNotification($trade_id, [
                    'type' => 'trade_update',
                    'trade_id' => $trade_id,
                    'status' => 'active',
                    'message' => 'Trade has been accepted and escrow funded',
                ]);
            }
        }
    } catch (Exception $notif_error) {
        // Log notification error but don't fail the request
        error_log("Notification error (non-fatal): " . $notif_error->getMessage());
    }
    
    echo json_encode([
        "success" => true,
        "message" => "Trade accepted and escrow funded successfully"
    ]);
} catch (Exception $e) {
    $conn->rollback();
    error_log("Accept trade error: " . $e->getMessage() . " | File: " . $e->getFile() . " | Line: " . $e->getLine());
    http_response_code(500);
    echo json_encode([
        "success" => false,
        "error" => "Failed to accept trade: " . $e->getMessage()
    ]);
} catch (Error $e) {
    $conn->rollback();
    error_log("Accept trade fatal error: " . $e->getMessage() . " | File: " . $e->getFile() . " | Line: " . $e->getLine());
    http_response_code(500);
    echo json_encode([
        "success" => false,
        "error" => "Failed to accept trade: " . $e->getMessage()
    ]);
}

if (isset($conn) && $conn) {
    $conn->close();
}
if (isset($transactions_conn) && $transactions_conn && !$transactions_conn->connect_error) {
    $transactions_conn->close();
}
?>

